﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Xml.Serialization;

namespace Minen
{
    /// <summary>
    /// enthält alle Spielrelevanten Daten und Funktionen
    /// </summary>
    public class Minensucher
    {
        Highscores scores;

        public Highscores Scores
        {
            get { return scores; }
        }

        GameData[] standardeinstellungen;

        public GameData[] Standardeinstellungen
        {
            get { return standardeinstellungen; }
        }

        GameData currenteinstellungen;

        public GameData Currenteinstellungen
        {
            get { return currenteinstellungen; }
            set { currenteinstellungen = value; }
        }

        feldstatus[][] spielfeld;

        //für Gewinnbedingung
        int counthiddencells;

        public int Counthiddencells
        {
            get { return counthiddencells; }
        }

        /// <summary>
        /// legt Standardeinstellungen fest und lädt Highscores und letzte Einstellungen
        /// </summary>
        /// <param name="path">hier wird später nur der Programmordner eingesetzt</param>
        public Minensucher(string path)
        {
            standardeinstellungen = new GameData[3];
            standardeinstellungen[0] = new GameData(9, 9, 12);
            standardeinstellungen[1] = new GameData(12, 12, 30);
            standardeinstellungen[2] = new GameData(15, 15, 50);

            currenteinstellungen = new GameData();
            scores = new Highscores();

            LoadHighscores(path);
            LoadLastGameData(path);         
        }

        #region Spielfeld erzeugen
        
        /// <summary>
        /// initialisiert Daten des Felds, d.h. zufällige Minenverteilung, Anfangsstatus für Felder
        /// </summary>
        /// <param name="gd">Einstellungen nach denen das Feld generiert wird</param>
        public void InitializeField(GameData gd)
        {
            spielfeld = new feldstatus[gd.Width][];
            counthiddencells = gd.Width * gd.Height;

            for (int i = 0; i < gd.Width; ++i)
            {
                spielfeld[i] = new feldstatus[gd.Height];
                for (int j = 0; j < gd.Height; ++j)
                {
                    spielfeld[i][j] = new feldstatus();
                    spielfeld[i][j].Clickable = true;
                    spielfeld[i][j].Status = 0;
                    spielfeld[i][j].Marked = 0;
                    spielfeld[i][j].Show = false;
                }
            }
            Random r = new Random();
            for (int i = 0; i < gd.Mines; ++i)
            {
                int x = r.Next(gd.Width);
                int y = r.Next(gd.Height);

                if (spielfeld[x][y].Status != -1) spielfeld[x][y].Status = -1;
                else --i;
            }

            for (int i = 0; i < gd.Width; ++i)
                for (int j = 0; j < gd.Height; ++j)
                {
                    if (spielfeld[i][j].Status != -1)
                        spielfeld[i][j].Status = CountNeighbours(gd, i, j);
                }
        }

        #endregion 

        #region Laden und Speichern der Spielvariablen

        /// <summary>
        /// letzte Spieldaten laden, falls keine gefunden, Standardeinstellungen verwenden
        /// </summary>
        /// <param name="path"></param>
        void LoadLastGameData(string path)
        {
            FileStream fs = null;
            XmlSerializer xmls = new XmlSerializer(typeof(GameData));

            try
            {
                fs = new FileStream(path + "\\UserGameData.mydata", FileMode.Open, FileAccess.Read);
                currenteinstellungen = xmls.Deserialize(fs) as GameData;
            }
            catch (Exception)
            {
                currenteinstellungen = standardeinstellungen[0];
                string s = "Keine Spieldaten gefunden, Standardeinstellung für Anfänger geladen.";
                System.Windows.Forms.MessageBox.Show(s);
            }
            finally
            {
                if (fs != null)
                    fs.Close();
            } 
        }

        /// <summary>
        /// speichert aktuelle Daten, wird beim Beenden aufgerufen
        /// </summary>
        /// <param name="path"></param>
        public void SaveCurrentGameData(string path)
        {
            FileStream fs = null;

            XmlSerializer xmls = new XmlSerializer(typeof(GameData));

            try
            {
                fs = new FileStream(path + "\\UserGameData.mydata", FileMode.Create, FileAccess.Write);
                xmls.Serialize(fs, currenteinstellungen);
            }
            catch (Exception e)
            {
                System.Windows.Forms.MessageBox.Show(e.InnerException.Message);

            }
            finally
            {
                if (fs != null)
                    fs.Close();
            }
        }

        /// <summary>
        /// lädt Highscores aus Datei, wenn nicht möglich, neue Standardhighscores erzeugen
        /// </summary>
        /// <param name="path"></param>
        void LoadHighscores(string path)
        {
            FileStream fs = null;

            XmlSerializer xmls = new XmlSerializer(typeof(Highscores));

            try
            {
                fs = new FileStream(path + "\\Highscores.score", FileMode.Open, FileAccess.Read);
                scores = xmls.Deserialize(fs) as Highscores;
            }
            catch (Exception)
            {
                scores.DefaultScores(); 
                string s = "Keine Highscores gefunden, Standardhighscores neu erstellt";
                System.Windows.Forms.MessageBox.Show(s);
            }
            finally
            {
                if (fs != null)
                    fs.Close();
            } 
        }

        /// <summary>
        /// speichert momentane Highscores
        /// </summary>
        /// <param name="path"></param>
        public void SaveHighscores(string path)
        {
            FileStream fs = null;

            XmlSerializer xmls = new XmlSerializer(typeof(Highscores));

            try
            {
                fs = new FileStream(path + "\\Highscores.score", FileMode.Create, FileAccess.Write);
                xmls.Serialize(fs, scores);
            }
            catch (Exception e)
            {
                System.Windows.Forms.MessageBox.Show(e.InnerException.Message);
            }
            finally
            {
                if (fs != null)
                    fs.Close();
            }
        }

        #endregion

        //zählt benachbarte Minen
        //in Felderstellung benötigt
        /// <summary>
        /// zählt die benachbarten Minen eines Feldes
        /// </summary>
        /// <param name="gd">Daten des Feldes, nötig für Randabfrage</param>
        /// <param name="x">Spalte</param>
        /// <param name="y">Zeile</param>
        /// <returns></returns>
        int CountNeighbours(GameData gd, int x, int y)
        {
            int neighbours = 0;
            for (int i = -1; i < 2; ++i)
            {
                for (int j = -1; j < 2; ++j)
                {
                    try
                    {
                        if (GetState(gd, x + i, y + j).Status == -1) neighbours++;
                    }
                    catch (IndexOutOfRangeException) { }
                }
            }
            return neighbours;

        }

        /// <summary>
        /// liest den feldstatus eines Feldes aus, incl. Abfangen von Randüberschreitungen
        /// </summary>
        /// <param name="gd">Daten des Feldes</param>
        /// <param name="x">Spalte</param>
        /// <param name="y">Zeile</param>
        /// <returns></returns>
        feldstatus GetState(GameData gd, int x, int y)
        {
            if (x < 0 || x >= gd.Width || y < 0 || y >= gd.Height)
                throw new IndexOutOfRangeException();
            return spielfeld[x][y];
        }

        /// <summary>
        /// wandelt feldstatus an gewünschten Koordinaten in String um
        /// </summary>
        /// <param name="x">Spalte</param>
        /// <param name="y">Zeile</param>
        /// <returns></returns>
        public string StateToString(int x, int y)
        {
            switch (spielfeld[x][y].Marked)
            {
                case 1: return "!";
                case 2: return "?";
            }

            if (spielfeld[x][y].Show == false) return "";

            switch (spielfeld[x][y].Status)
            {
                case 0: return "";
                case -1: return "M";
                default: return spielfeld[x][y].Status.ToString();
            }
        }

        #region Funktionen die bei Klick aufs Feld aufgerufen werden

        /// <summary>
        /// nächsten Marker setzen falls möglich, Ergebnis zurückgeben
        /// </summary>
        /// <param name="x">Spalte</param>
        /// <param name="y">Zeile</param>
        /// <returns></returns>
        public int NextMark(int x, int y)
        {
            int mark = spielfeld[x][y].Marked;
            switch (mark)
            {
                case 0:
                    spielfeld[x][y].Marked = 1;
                    spielfeld[x][y].Clickable = false;
                    return 1;
                case 1:
                    spielfeld[x][y].Marked = 2;
                    return 2;
                case 2:
                    spielfeld[x][y].Marked = 0;
                    spielfeld[x][y].Clickable = true;
                    return 0;
            }
            return -1;
        }

        /// <summary>
        /// fragt ab ob eine Zelle bereits angeklickt wurde
        /// war nötig, damit keine Endlosschleife bei Clearzeroes in Form1 entsteht
        /// </summary>
        /// <param name="x">Spalte</param>
        /// <param name="y">Zeile</param>
        /// <returns></returns>
        public bool Show(int x, int y)
        {
            return spielfeld[x][y].Show;
        }

        /// <summary>
        /// gibt zurück ob anklickbar (nicht wenn bereits angezeigt oder markiert)
        /// </summary>
        /// <param name="x">Spalte</param>
        /// <param name="y">Zeile</param>
        /// <returns></returns>
        public bool Clickable(int x, int y)
        {
            return spielfeld[x][y].Clickable;
        }

        
        /// <summary>
        /// führt Klick auf eine Zelle aus
        ///Exception damit Clearzeroes einfacher zu schreiben ist
        ///(alles was Ränder überschreitet wird ignoriert)
        /// </summary>
        /// <param name="x">Spalte</param>
        /// <param name="y">Zeile</param>
        /// <returns></returns>
        public int Click(int x, int y)
        {
            if (x < 0 || x >= Currenteinstellungen.Width || y < 0 || y >= Currenteinstellungen.Height)
                throw new IndexOutOfRangeException();
            if (spielfeld[x][y].Clickable && !spielfeld[x][y].Show)
            {
                counthiddencells--;
                spielfeld[x][y].Clickable = false;
                spielfeld[x][y].Show = true;
            }
            return GetState(currenteinstellungen, x, y).Status;
        }

        /// <summary>
        /// keine Klicks mehr zulassen wenn Spiel zu Ende
        /// </summary>
        public void Disable()
        {
            for (int i = 0; i < currenteinstellungen.Width; ++i)
                for (int j = 0; j < currenteinstellungen.Height; ++j)
                    spielfeld[i][j].Clickable = false;
        }

        #endregion
    }
}
