﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using SdlDotNet.Graphics;
using SdlDotNet.Audio;

namespace Pong
{
    /// <summary>
    /// Der Ball im Spiel von Pong
    /// </summary>
    class Ball : PongItem
    {
        #region Variablendeklarierung

        /// <summary>
        /// Geschwindigkeit des Balls in x-Richtung
        /// </summary>
        double ballSpeedX;

        /// <summary>
        /// Geschwindigkeit des Balls in y-Richtung
        /// </summary>
        double ballSpeedY;

        /// <summary>
        /// Startgeschwindigkeit des Balls in x-Richtung
        /// </summary>
        double startBallSpeedX;

        /// <summary>
        /// Startgeschwindigkeit des Balls in y-Richtung
        /// </summary>
        double startBallSpeedY;

        /// <summary>
        /// Maximale Geschwindigkeit in x-Richtung
        /// </summary>
        double maxSpeedX;

        /// <summary>
        /// Geschwindigkeitserhöhung des Balls, wenn ein Paddle getroffen wird
        /// </summary>
        double deltaSpeedX;

        /// <summary>
        /// Legt die momentane Zugehörigkeit des Balls fest
        /// </summary>
        Spieler.Seite seit;

        /// <summary>
        /// Die Balleinstellung dieses Balls
        /// </summary>
        BallEinstellung balleinstell;

        /// <summary>
        /// Der Sound, der beim Treffen eines Paddles gespielt wird
        /// </summary>
        Sound paddletreffer;

        /// <summary>
        /// Der Sound, der beim Rausfliegen des Balls gespielt wird
        /// </summary>
        Sound ballraus;

        /// <summary>
        /// Der Sound, der beim Treffen der oberen oder unteren Wand gespielt wird
        /// </summary>
        Sound wandtreffen;

        /// <summary>
        /// Legt fest, ob der Ball im Spiel aktiv ist (für BallVerdoppelItem)
        /// </summary>
        bool aktiv;

        #endregion

        #region Konstruktor

        /// <summary>
        /// Erstellt einen neuen Ball
        /// </summary>
        /// <param name="seite"></param>
        /// <param name="balleinstellung"></param>
        public Ball(Spieler.Seite seite, BallEinstellung balleinstellung)
            : base("Ball.png")
        {
            this.seit = seite;
            this.balleinstell = balleinstellung;

            // Die Geschwindigkeiten je nach Balleinstellung einstellen
            ballEinstellen();

            // Wenn der Ball auf der rechten Seite startet, muss er nach links fliegen
            if (seite == Spieler.Seite.Rechts)
                ballSpeedX *= -1;

            // Sounds initialisieren
            paddletreffer = new Sound("Paddletreffer.wav");
            ballraus = new Sound("Ballraus.wav");
            wandtreffen = new Sound("Wandtreffen.wav");

            aktiv = true;
        }

        #endregion

        #region Variablenzugriff

        /// <summary>
        /// Setzt die Geschwindigkeit des Balls in x-Richtung  oder gibt diese zurück
        /// </summary>
        public double BallSpeedX
        {
            get { return ballSpeedX; }
            set { ballSpeedX = value; }
        }

        /// <summary>
        /// Setzt die Geschwindigkeit des Balls in y-Richtung  oder gibt diese zurück
        /// </summary>
        public double BallSpeedY
        {
            get { return ballSpeedY; }
            set { ballSpeedY = value; }
        }

        /// <summary>
        /// Gibt die Balleinstellung des Balls zurück
        /// </summary>
        public BallEinstellung Balleinstell
        {
            get { return balleinstell; }
        }

        /// <summary>
        /// Setzt die Aktivität des Balls oder gibt diese zurück
        /// </summary>
        public bool Aktiv
        {
            get { return aktiv; }
            set { aktiv = value; }
        }

        /// <summary>
        /// Setzt die aktuelle Seite des Balls oder gibt diese zurück
        /// </summary>
        public Spieler.Seite Seit
        {
            get { return seit; }
            set { seit = value; }
        }

        #endregion

        #region Methoden

        /// <summary>
        /// Stellt fest, ob sich die Sprites vom Ball und pi überschneiden
        /// </summary>
        /// <param name="pi"></param>
        /// <returns></returns>
        private bool intersectsWith(PongItem pi)
        {
            return sprite.IntersectsWith(new Rectangle((int)pi.X, (int)pi.Y, pi.Width, pi.Height));
        }


        /// <summary>
        /// Zeichnet den Sprite des Balls und bewegt diesen
        /// </summary>
        public override void draw()
        {
            // Nur zeichnen, wenn aktiv
            if (aktiv)
            {
                // Überprüfen, ob ein Paddle oder ein EventItem getroffen wurde
                foreach (PongItem pi in Pongmaster.Items)
                {
                    if (pi is Spieler && intersectsWith(pi))
                    {
                        //Ballzugehörigkeit festlegen
                        seit = (pi as Spieler).Seit;

                        //Ball dreht in die Gegenrichtung um
                        ballSpeedX *= -1;

                        //Ballgeschwindigkeit erhöhen, solange die Maximalgeschwindigkeit noch nicht erreicht ist
                        if (Math.Abs(BallSpeedX) < maxSpeedX)
                        {
                            if (BallSpeedX < 0)
                                ballSpeedX -= deltaSpeedX;
                            else
                                ballSpeedX += deltaSpeedX;
                        }

                        // y-Geschwindigkeit des Balls soll sich je nach Auftreffpunkt auf dem Paddle verändern (-3 ganz oben, +3 ganz unten)
                        ballSpeedY += 6 / ((double)pi.Height + (double)Height) * (Y + (double)Height / 2) - 3 - 6 / ((double)pi.Height + (double)Height) * (pi.Y - (double)Height / 2);

                        paddletreffer.Play();
                    }
                    else if (pi is EventItem && intersectsWith(pi) && (pi as EventItem).Aktiv)
                    {
                        (pi as EventItem).getroffen(seit, this);
                    }
                }

                //Überprüfen, ob eine obere oder untere Wand getroffen wurde
                if (Y < 0 || (Y + Height) > Video.Screen.Height)
                {
                    //Ball dreht in die Gegenrichtung um
                    ballSpeedY *= -1;

                    wandtreffen.Play();
                }

                //Ball bewegt sich weiter
                X += BallSpeedX;
                Y += ballSpeedY;

                //Überprüfen ob der Ball links oder Rechts aus dem Bildschirm fliegt
                if (X <= 0)
                {
                    foreach (PongItem pi in Pongmaster.Items)
                    {
                        if (pi is Spieler && (pi as Spieler).Seit == Spieler.Seite.Rechts)
                        {
                            //Punkte des rechten Spielers erhöhen
                            (pi as Spieler).Punkte += 1;

                            // Ball soll vor dem rechten Paddle starten
                            X = Video.Screen.Width - pi.Width - Width - 1;
                            Y = pi.Y + pi.Height / 2;

                            // Geschwindigkeiten auf Startwert festlegen
                            ballSpeedX = -startBallSpeedX;
                            ballSpeedY = startBallSpeedY;

                            //Ballzugehörigkeit festlegen
                            seit = (pi as Spieler).Seit;
                        }
                    }

                    ballraus.Play();
                }
                else if ((X + Width) >= Video.Screen.Width)
                {
                    foreach (PongItem pi in Pongmaster.Items)
                    {
                        if (pi is Spieler && (pi as Spieler).Seit == Spieler.Seite.Links)
                        {
                            //Punkte des linken Spielers erhöhen
                            (pi as Spieler).Punkte += 1;

                            // Ball soll vor dem linken Paddle starten
                            X = pi.Width + 1;
                            Y = pi.Y + pi.Height / 2;

                            // Geschwindigkeiten auf Startwert festlegen
                            ballSpeedX = startBallSpeedX;
                            ballSpeedY = startBallSpeedY;

                            //Ballzugehörigkeit festlegen
                            seit = (pi as Spieler).Seit;
                        }
                    }

                    ballraus.Play();
                }

                // Neuzeichnen
                base.draw();
            }
        }

        /// <summary>
        /// Stellt die Geschwindigkeiten des Balls auf die jeweilige Balleinstellung
        /// </summary>
        public void ballEinstellen()
        {
            if (balleinstell == BallEinstellung.Amateur)
            {
                // Standardstartgeschwindigkeiten des Balls
                startBallSpeedX = 3.0;
                startBallSpeedY = 1;

                //Standardmaximalgeschwindigkeit des Balls in x-Richtung
                maxSpeedX = 6.0;

                //Standardgeschwindigkeitsänderung des Balls in x-Richtung
                deltaSpeedX = 0.3;

                // Geschwindigkeiten auf Startwert festlegen
                ballSpeedX = startBallSpeedX;
                ballSpeedY = startBallSpeedY;
            }
            else if (balleinstell == BallEinstellung.Fortgeschritten)
            {
                // Standardstartgeschwindigkeiten des Balls
                startBallSpeedX = 5.0;
                startBallSpeedY = 1.6;

                //Standardmaximalgeschwindigkeit des Balls in x-Richtung
                maxSpeedX = 10.0;

                //Standardgeschwindigkeitsänderung des Balls in x-Richtung
                deltaSpeedX = 0.5;

                // Geschwindigkeiten auf Startwert festlegen
                ballSpeedX = startBallSpeedX;
                ballSpeedY = startBallSpeedY;
            }
            else if (balleinstell == BallEinstellung.Profi)
            {
                // Standardstartgeschwindigkeiten des Balls
                startBallSpeedX = 7.0;
                startBallSpeedY = 2.3;

                //Standardmaximalgeschwindigkeit des Balls in x-Richtung
                maxSpeedX = 14.0;

                //Standardgeschwindigkeitsänderung des Balls in x-Richtung
                deltaSpeedX = 0.7;

                // Geschwindigkeiten auf Startwert festlegen
                ballSpeedX = startBallSpeedX;
                ballSpeedY = startBallSpeedY;
            }
        }

        #endregion

        #region Enumeration BallEinstellung

        /// <summary>
        /// Enumeration für die verschiedenen Geschwindigkeitseinstellungen des Balls
        /// </summary>
        public enum BallEinstellung : int
        {
            /// <summary>
            /// Langsame Ballgeschwindigkeit
            /// </summary>
            Amateur = 0,

            /// <summary>
            /// Mittlere Ballgeschwindigkeit
            /// </summary>
            Fortgeschritten = 1,

            /// <summary>
            /// Schnelle Ballgeschwindigkeit
            /// </summary>
            Profi = 2,
        }

        #endregion
    }
}
