﻿using System;
using System.IO;
using System.Drawing;

using SdlDotNet;
using SdlDotNet.Graphics;
using SdlDotNet.Graphics.Sprites;
using SdlDotNet.Core;
using SdlDotNet.Input;
using System.Collections.Generic;


namespace Pong
{
    /// <summary>
    /// Die Verwaltungsklasse für das Spiel, hier laufen alle Fäden zusammen
    /// </summary>
    public class Pong
    {
        #region Variablendeklarierung

        /// <summary>
        /// Legt fest, ob das Spiel aktiv ist (true) oder das Menü (false)
        /// </summary>
        bool pongAktiv;

        /// <summary>
        /// PongMaster, der alle relevanten PongItems enthält und diese entsprechend behandelt
        /// </summary>
        PongMaster pongmaster;

        /// <summary>
        /// Hauptmenü, das die Menüaufgaben übernimmt
        /// </summary>
        HauptMenu hauptmenu;

        /// <summary>
        /// Alle Optionen, die der Spieler einstellen kann
        /// </summary>
        Option[] optionen;

        /// <summary>
        /// Zufallszahlengenerator für die Positionierung der EventItems
        /// </summary>
        Random random;
        
        /// <summary>
        /// Alle EventItems, die das Spiel zu bieten hat
        /// </summary>
        EventItem[] eventitems;

        #endregion

        #region Konstruktor

        /// <summary>
        /// Erstellt ein neues Pong
        /// </summary>
        public Pong()
        {
            // Legt die Fenstergröße fest
            Video.SetVideoMode(800, 600);

            // Klicken auf das X soll Events_Quit aufrufen
            Events.Quit += new EventHandler<QuitEventArgs>(Events_Quit);

            // Jeder Frame soll Events_Tick aufrufen, wenn er gezeichnet wird
            Events.Tick += new EventHandler<TickEventArgs>(Events_Tick);

            // Drücken von Tasten soll Events_Keydown aufrufen
            Events.KeyboardDown += new EventHandler<KeyboardEventArgs>(Events_Keydown);

            // Loslassen von Tasten soll Events_Keyup aufrufen
            Events.KeyboardUp += new EventHandler<KeyboardEventArgs>(Events_Keyup);

            // Die Optionen festlegen
            optionen = new Option[Enum.GetValues(typeof(Optionen)).Length];
            optionen[(int)Optionen.NameRechterSpieler] = new TextOption("Name rechter Spieler", "RECHTER SPIELER");
            optionen[(int)Optionen.NameLinkerSpieler] = new TextOption("Name linker Spieler", "LINKER SPIELER");
            optionen[(int)Optionen.KIverwenden] = new AuswahlOption("Linkes Paddle durch KI steuern", Enum.GetNames(typeof(Pong.JaNein)), (int)Pong.JaNein.Nein);
            optionen[(int)Optionen.AnzahlBaelle] = new IntOption("Anzahl Bälle", 1, 1, 4);
            optionen[(int)Optionen.SiegPunkte] = new IntOption("Siegpunkte", 20, 10, 50);
            optionen[(int)Optionen.BallEinstellung] = new AuswahlOption("Ballverhalten", Enum.GetNames(typeof(Ball.BallEinstellung)), (int)Ball.BallEinstellung.Fortgeschritten);
            optionen[(int)Optionen.EventAuftreten] = new AuswahlOption("Itemhäufigkeit", Enum.GetNames(typeof(PongMaster.EventAuftreten)), (int)PongMaster.EventAuftreten.Normal);

            // Initialisierung des Randoms
            random = new Random();
            
            // EventItems festlegen
            eventitems = new EventItem[Enum.GetValues(typeof(EventItems)).Length];
            eventitems[(int)EventItems.PlusPunkt] = new PlusPunktItem(random);
            eventitems[(int)EventItems.MinusPunkt] = new MinusPunktItem(random);
            eventitems[(int)EventItems.BoostLinks] = new BoostLinksItem(random);
            eventitems[(int)EventItems.BoostRechts] = new BoostRechtsItem(random);
            eventitems[(int)EventItems.BoostOben] = new BoostObenItem(random);
            eventitems[(int)EventItems.BoostUnten] = new BoostUntenItem(random);
            eventitems[(int)EventItems.BallVerdopplung] = new BallVerdoppelItem(random);

            // Initialisierung
            init();
        }

        #endregion

        #region Methoden

        /// <summary>
        /// Initialisiert den PongMaster und das Hauptmenü und weist den Events die richtigen Methoden zu
        /// </summary>
        public void init()
        {
            // Das Hauptmenü soll am Anfang angezeigt werden
            pongAktiv = false;

            // Neue Instanzen von PongMaster und HauptMenu werden erstellt
            pongmaster = new PongMaster(optionen,eventitems);
            hauptmenu = new HauptMenu(optionen,eventitems);

            // Den PongMasterEvents werden die entsprechenden Methoden zugewiesen
            pongmaster.spielbeender += new PongMaster.SpielBeender(spielbeenden);

            // Den HauptmenüEvents werden die entsprechenden Methoden zugewiesen
            hauptmenu.spielstarter += new HauptMenu.SpielStarter(spielstarten);
            hauptmenu.beender += new HauptMenu.Beender(beenden);
        }

        /// <summary>
        /// Wird beim Klicken auf das X ausgeführt
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Events_Quit(object sender, QuitEventArgs e)
        {
            beenden();
        }

        /// <summary>
        /// Wird beim Zeichnen jedes Frames aufgerufen
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Events_Tick(object sender, TickEventArgs e)
        {
            if (pongAktiv)
                pongmaster.Events_Tick(sender, e);
            else
                hauptmenu.Events_Tick(sender, e);
        }


        /// <summary>
        /// Wird bei jedem Drücken einer Taste aufgerufen
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Events_Keydown(object sender, KeyboardEventArgs e)
        {
            if (pongAktiv)
                pongmaster.Events_Keydown(sender, e);
            else
                hauptmenu.Events_Keydown(sender, e);
        }


        /// <summary>
        /// Wird bei jedem Loslassen einer Taste aufgerufen
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Events_Keyup(object sender, KeyboardEventArgs e)
        {
            if (pongAktiv)
                pongmaster.Events_Keyup(sender, e);
        }

        /// <summary>
        /// Startet ein neues Spiel
        /// </summary>
        public void spielstarten()
        {
            // Der Pongmaster wird initialisert und die Events werden zu ihm umgeleitet
            pongmaster.init();
            pongAktiv = true;
        }

        /// <summary>
        /// Beendet ein Spiel
        /// </summary>
        public void spielbeenden()
        {
            pongAktiv = false;
        }

        /// <summary>
        /// Beendet das Programm
        /// </summary>
        private void beenden()
        {
            Events.QuitApplication();
        }

        /// <summary>
        /// Erweckt das Programm zum Leben
        /// </summary>
        public void Go()
        {
            Events.Run();
        }

        #endregion

        #region Main-Methode

        /// <summary>
        /// Main-Methode des Programms
        /// </summary>
        [STAThread]
        public static void Main()
        {
            Pong pong = new Pong();
            pong.Go();
        }

        #endregion

        #region Enumeration JaNein

        /// <summary>
        /// Enumeration für das Abfragen von Ja/Nein-Optionen
        /// </summary>
        public enum JaNein : int
        {
            /// <summary>
            /// Ja
            /// </summary>
            Ja = 0,

            /// <summary>
            /// Nein
            /// </summary>
            Nein = 1,

        }

        #endregion

        #region Enumeration Optionen

        /// <summary>
        /// Enumeration für die Optionen
        /// </summary>
        public enum Optionen : int
        {
            /// <summary>
            /// Index für den Namen des rechten Spielers
            /// </summary>
            NameRechterSpieler = 0,

            /// <summary>
            /// Index für den Namen des linken Spielers
            /// </summary>
            NameLinkerSpieler = 1,

            /// <summary>
            /// Index für die das Verwenden der KI
            /// </summary>
            KIverwenden = 2,

            /// <summary>
            /// Index für die Anzahl der Bälle im Spiel
            /// </summary>
            AnzahlBaelle = 3,

            /// <summary>
            /// Index für die Punkte, die zum Sieg benötigt werden
            /// </summary>
            SiegPunkte = 4,

            /// <summary>
            /// Index für die BallEinstellung
            /// </summary>
            BallEinstellung = 5,
            
            /// <summary>
            /// Index für das EventAuftreten
            /// </summary>
            EventAuftreten = 6,
        }

        #endregion

        #region Enumeration EventItems

        /// <summary>
        /// Enumeration für die EventItems
        /// </summary>
        public enum EventItems : int
        {
            /// <summary>
            /// Index für das PlusPunktItem
            /// </summary>
            PlusPunkt = 0,

            /// <summary>
            /// Index für das MinusPunktItem
            /// </summary>
            MinusPunkt = 1,

            /// <summary>
            /// Index für das BoostLinksItem
            /// </summary>
            BoostLinks = 2,

            /// <summary>
            /// Index für das BoostRechtsItem
            /// </summary>
            BoostRechts = 3,

            /// <summary>
            /// Index für das BoostObenItem
            /// </summary>
            BoostOben = 4,

            /// <summary>
            /// Index für das BoostUntenItem
            /// </summary>
            BoostUnten = 5,

            /// <summary>
            /// Index für das BallVerdoppelItem
            /// </summary>
            BallVerdopplung = 6,
        }

        #endregion
    }
}
