﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.ComponentModel;
using System.Collections;

///
/// WRITTEN BY FLORIAN RAPPL, 2011
/// CODE IS FREE TO USE -- CPOL [ Code Project Open License ]
/// 

namespace rsi.Controls.iToolTip
{
    [ProvideProperty("ToolTip", typeof(Control))]
    [ToolboxItemFilter("System.Windows.Forms")]
    public class InternetToolTip : Component, IExtenderProvider
    {
        #region members

        IToolTipDataProvider dataProvider;
        IToolTipView dataView;
        bool async;
        bool loading;
        string loadText;
        Hashtable collection;
        Control active;

        #endregion

        #region ctor

        public InternetToolTip()
        {
            collection = new Hashtable();
            async = true;
            dataProvider = new StringDataProvider();
            dataView = new BasicToolTipView();
            loadText = string.Empty;
        }

        #endregion

        #region public methods

        /// <summary>
        /// Sets the ToolTip (query) for a specific control.
        /// </summary>
        /// <param name="control">The control where the ToolTip should be displayed for</param>
        /// <param name="caption">The ToolTip (StringDataProvider) or ToolTip query in general</param>
        public void SetToolTip(Control control, string caption)
        {
            if (caption.Equals(string.Empty))
            {
                if (collection.ContainsKey(control))
                {
                    control.MouseEnter -= new EventHandler(control_MouseEnter);
                    control.MouseLeave -= new EventHandler(control_MouseLeave);
                    collection.Remove(control);
                }

                return;
            }

            if (collection.ContainsKey(control))
                collection[control] = caption;
            else
            {
                collection.Add(control, caption);
                control.MouseEnter += new EventHandler(control_MouseEnter);
                control.MouseLeave += new EventHandler(control_MouseLeave);
            }
        }

        /// <summary>
        /// Gets the ToolTip (query) for a specific control
        /// </summary>
        /// <param name="control">The control where the ToolTip is displayed for</param>
        /// <returns>The ToolTip (StringDataProvider) or ToolTip query in general</returns>
        public string GetToolTip(Control control)
        {
            if (collection.ContainsKey(control))
                return collection[control].ToString();

            return string.Empty;
        }

        #endregion

        #region private methods

        void control_MouseLeave(object sender, EventArgs e)
        {
            dataView.HideToolTip();

            if (loading)
            {
                dataProvider.CancelDataAsync(sender as Control);

                if ((sender as Control).Equals(active))
                    loading = false;
            }
        }

        void control_MouseEnter(object sender, EventArgs e)
        {
            loading = async;
            Control c = sender as Control;
            active = c;
            Point p = c.Parent.PointToScreen(c.Location);
            dataView.ShowToolTip(p, c.Size);

            if (async)
            {
                dataView.DrawLoading(loadText);
                dataProvider.RequestDataAsync(collection[c].ToString(), c);
            }
            else
            {
                try
                {
                    dataView.DrawToolTip(dataProvider.RequestData(collection[c].ToString()));
                }
                catch (Exception ex)
                {
                    dataView.DrawException(ex);
                }
            }
        }

        void OnRequest(object sender, ToolTipDataEventArgs e)
        {
            if (!e.Control.Equals(active))
                return;

            loading = false;

            if (e.Success)
                dataView.DrawToolTip(e.Result);
            else
                dataView.DrawException(e.Exception);
        }

        #endregion

        #region properties

        /// <summary>
        /// Gets or sets if the data should be requested asynchronously.
        /// </summary>
        [Description("Determines if the data provider is used asynchronously.")]
        [DefaultValue(true)]
        public bool Async
        {
            get { return async; }
            set { async = value; }
        }

        /// <summary>
        /// Gets or sets the data provider that will be used for requesting data.
        /// </summary>
        [Description("The data provider that is used for requesting tooltips.")]
        public IToolTipDataProvider DataProvider
        {
            get { return dataProvider ?? new StringDataProvider(); }
            set 
            { 
                dataProvider = value;
                dataProvider.RequestDataHandled += new ToolTipDataEventHandler(OnRequest);
            }
        }

        /// <summary>
        /// Gets or sets the data view that will be used for showing the tooltip.
        /// </summary>
        [Description("The view provider that is used for displaying tooltips.")]
        public IToolTipView DataView
        {
            get { return dataView; }
            set { dataView = value; }
        }

        /// <summary>
        /// Gets or sets the text that will be displayed on asynchronous requests.
        /// </summary>
        [Description("This text will be shown on asynchronous requests.")]
        public string LoadText
        {
            get { return loadText; }
            set { loadText = value; }
        }

        #endregion

        #region IExtenderProvider Member

        public bool CanExtend(object extendee)
        {
            return (extendee is Control && !(extendee is Form) && !(extendee is InternetToolTip));
        }

        #endregion
    }
}
