﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SdlDotNet.Core;
using System.Drawing;
using System.Reflection;

namespace TowerDefence
{
    public class TowerBase : BaseObject
    {
        /// <summary>
        /// in units of the maze lattice spacing
        /// </summary>
        [DescriptorProperty]
        public virtual float Range { get { return 3f; } }

        [DescriptorProperty]
        public virtual decimal Cost { get { return Math.Round((decimal)(Rating / 6) + 1M); } }

        private int TimeToNextShot { get; set; }

        protected List<UnitBase> targets;

        public virtual List<UnitBase> Targets
        {
            get { return targets.Where(ub => ub.Strategy != MazeStrategy.Air).ToList(); }//standard towers shoot only on non-flying objects
            set { targets = value; }
        }

        public class ShotEventArgs : EventArgs
        {
            public ShotBase Shot { get; private set; }
            public ShotEventArgs(ShotBase shot)
            {
                Shot = shot;
            }
        }

        public event EventHandler<ShotEventArgs> Shot;

        [DescriptorProperty]
        public virtual ShotTypeDescriptor ShotType { get { return ShotTypeDescriptor.GetDescriptor(typeof(ShotBase)); } }

        public TowerBase()
        {
            TimeToNextShot = 0;
            Direction = TowerDefence.Direction.Left;
        }

        public override void Update(TickEventArgs args)
        {
            base.Update(args);
            ShotBase shot = null;
            if (TimeToNextShot <= 0)
                shot = Shoot();
            if (shot != null)
            {
                if (Shot != null)
                    Shot(this, new ShotEventArgs(shot));
                TimeToNextShot = (int)(1 / (Speed / 1000));
            }
            else
                TimeToNextShot -= args.TicksElapsed;
        }

        public virtual ShotBase Shoot()
        {
            var nearestTarget = Targets.OrderBy(u => DistanceTo(u.MazeCoordinates)).FirstOrDefault();
            if (nearestTarget != null && DistanceTo(nearestTarget.MazeCoordinates) < Range)
            {
                ShotBase shot = Assembly.GetExecutingAssembly().CreateInstance(ShotType.Type.FullName) as ShotBase;
                shot.MazeCoordinates = MazeCoordinates;
                shot.DirectionOfMotion = nearestTarget.MazeCoordinates.Subtract(MazeCoordinates);
                shot.Direction = shot.DirectionOfMotion.ToDirection();
                this.Direction = shot.Direction;
                shot.Targets = Targets;
                return shot;
            }
            else
                return null;
        }

        [DescriptorProperty]
        public override float Speed { get { return 1f; } }

        public override string Description { get { return @"base of all towers - low budget defensor"; } }

        protected new static List<Animation.Animation> animations;

        public override double Rating
        {
            get { return Speed * Math.Log(Range + 1) * (double)ShotType.Rating; }
        }
    }
}
