﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Drawing;
using SdlDotNet.Particles;
using SdlDotNet.Core;
using System.Reflection;

namespace TowerDefence
{
    /// <summary>
    /// fancy functions for faster and more comfortable programming
    /// </summary>
    public static class Helper
    {
        public static string FindFile(string filename, string startDirectory = null)
        {
            if (startDirectory == null)
                startDirectory = Directory.GetCurrentDirectory();

            //look for the file down the hirarchy
            string fullName = Directory.EnumerateFiles(startDirectory, filename, SearchOption.AllDirectories).FirstOrDefault();
            if (fullName != null)
                return fullName;

            //look for the file up the hirarchy
            return LookElsewhere(filename, startDirectory);
        }

        private static string LookElsewhere(string filename, string startDirectory)
        {
            DirectoryInfo parent = null;
            try
            {
                parent = Directory.GetParent(startDirectory);

                string fullName;
                //look for the file on the same hirarchy level
                DirectoryInfo current = new DirectoryInfo(startDirectory);
                foreach (var item in parent.EnumerateDirectories())
                {
                    if (item.FullName != current.FullName)
                    {
                        fullName = Directory.EnumerateFiles(startDirectory, filename, SearchOption.AllDirectories).FirstOrDefault();
                        if (fullName != null)
                            return fullName;
                    }
                }

            }
            catch (Exception)
            {
                throw new FileNotFoundException("While looking for file " + filename + " in method Helper.LookElsewhere");
            }

            //look for the file up the hirarchy
            return LookElsewhere(filename, parent.FullName);
        }

        static Random rng = new Random();

        public static float Square(this PointF point)
        {
            return point.X * point.X + point.Y * point.Y;
        }

        public static float Norm(this PointF point)
        {
            return (float)Math.Sqrt(point.X * point.X + point.Y * point.Y);
        }

        public static PointF Add(this PointF P1, PointF P2)
        {
            return new PointF(P1.X + P2.X, P1.Y + P2.Y);
        }

        public static PointF Subtract(this PointF P1, PointF P2)
        {
            return new PointF(P1.X - P2.X, P1.Y - P2.Y);
        }

        public static float ProjectOn(this PointF P1, PointF P2)
        {
            return P1.X * P2.X + P1.Y * P2.Y;
        }

        public static PointF Scale(this PointF point, float factor)
        {
            return new PointF(point.X * factor, point.Y * factor);
        }

        public static PointF Normalize(this PointF point)
        {
            float normFactor = 1 / point.Norm();
            return new PointF(point.X * normFactor, point.Y * normFactor);
        }

        public static PointF Position(this ParticleCircle particleCircle)
        {
            return new PointF(particleCircle.X, particleCircle.Y);
        }

        public static ParticleCircle Position(this ParticleCircle particleCircle, PointF point)
        {
            particleCircle.X = point.X;
            particleCircle.Y = point.Y;
            return particleCircle;
        }

        public static PointF Subtract(this Vector P1, PointF P2)
        {
            return Subtract(new PointF(P1.X, P1.Y), P2);
        }

        public static PointF Subtract(this PointF P1, Vector P2)
        {
            return Subtract(P1, new PointF(P2.X, P2.Y));
        }

        public static float ProjectOn(this Vector P1, PointF P2)
        {
            return ProjectOn(new PointF(P1.X, P1.Y), P2);
        }

        public static Vector ToVector(this PointF point)
        {
            return new Vector(point.X, point.Y, 0.0f);
        }

        public static Vector Scale(this Vector point, float factor)
        {
            return new Vector(point.X * factor, point.Y * factor, point.Z * factor);
        }

        public static float ProjectOn(this Vector P1, Vector P2)
        {
            return P1.X * P2.X + P1.Y * P2.Y + P1.Z * P2.Z;
        }

        public static Vector Add(this Vector P1, PointF P2)
        {
            return new Vector(P1.X + P2.X, P1.Y + P2.Y, P1.Z);
        }

        public static Vector Add(this Vector P1, Vector P2)
        {
            return new Vector(P1.X + P2.X, P1.Y + P2.Y, P1.Z + P2.Z);
        }

        public static float Norm(this Vector point)
        {
            return (float)Math.Sqrt(point.X * point.X + point.Y * point.Y + point.Z * point.Z);
        }

        public static PointF Regularize(this PointF point)
        {
            if (point.X == 0 && point.Y == 0)
                return new Point(rng.Next(2) * 2 - 1, rng.Next(2) * 2 - 1);
            else
                return point;
        }

        public static Point Subtract(this Point P1, Point P2)
        {
            return new Point(P1.X - P2.X, P1.Y - P2.Y);
        }

        public static PointF Scale(this Point point, float factor)
        {
            return new PointF(point.X * factor, point.Y * factor);
        }

        public static float Norm(this Point point)
        {
            return (float)Math.Sqrt(point.X * point.X + point.Y * point.Y);
        }

        public static PointF ToPointF(this Point point)
        {
            return new PointF(point.X, point.Y);
        }

        public static Point ToPoint(this PointF point)
        {
            return new Point((int)Math.Round((decimal)point.X), (int)Math.Round((decimal)point.Y));
        }

        public static PointF Normalize(this Point point)
        {
            float normFactor = 1 / point.Norm();
            return new PointF(point.X * normFactor, point.Y * normFactor);
        }

        public static Point Add(this Point P1, Point P2)
        {
            return new Point(P1.X + P2.X, P1.Y + P2.Y);
        }

        public static PointF Add(this Point P1, PointF P2)
        {
            return new PointF(P1.X + P2.X, P1.Y + P2.Y);
        }

        public static PointF Scale(this Point point, SizeF factor)
        {
            return new PointF(point.X * factor.Width, point.Y * factor.Height);
        }

        public static PointF Scale(this PointF point, SizeF factor)
        {
            return new PointF(point.X * factor.Width, point.Y * factor.Height);
        }

        public static SizeF Divide(this Size S1, Size S2)
        {
            return new SizeF(S1.Width / (float)S2.Width, S1.Height / (float)S2.Height);
        }

        public static Point Rescale(this Point point, Size oldSize, Size newSize)
        {
            return point.Scale(newSize.Divide(oldSize)).ToPoint();
        }

        public static PointF Rescale(this PointF point, Size oldSize, Size newSize)
        {
            return point.Scale(newSize.Divide(oldSize)).ToPoint();
        }

        public static IEnumerable<Type> EnumerateDerivedTypes(this Type BaseType, bool IncludeBaseType = false)
        {
            IEnumerable<Type> derivedTypes = System.Reflection.Assembly.GetExecutingAssembly().GetTypes().Where(t => t.IsSubclassOf(BaseType));
            if (IncludeBaseType)
            {
                List<Type> baseTypeList = new List<Type>();
                baseTypeList.Add(BaseType);
                return derivedTypes.Union(baseTypeList);
            }
            else
            {
                return derivedTypes;
            }
        }

        public static Direction ToDirection(this PointF direction)
        {
            if (direction.X > direction.Y)//top,right
            {
                if (direction.X > -direction.Y)//bottom,right
                    return Direction.Right;
                else
                    return Direction.Top;
            }
            else//bottom,left
            {
                if (direction.X > -direction.Y)//bottom,right
                    return Direction.Bottom;
                else
                    return Direction.Left;
            }
        }

        //get (hiding, "new", static) field via Reflection (workaround for static override)
        public static FieldInfo getNewField(this Type t, string fieldName)
        {
            FieldInfo newField = t.GetField(fieldName, BindingFlags.NonPublic | BindingFlags.Static);
            while (newField == null && t.BaseType != typeof(object))
            {
                t = t.BaseType;
                newField = t.GetField(fieldName, BindingFlags.NonPublic | BindingFlags.Static);
            }
            return newField;
        }
    }
}
