﻿using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using System.Drawing;
using System;

namespace TowerDefence
{
    /// <summary>
    /// ListBox specialised to show all possible objects in the game
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class ObjectListBox : ListBox
    {
        public TypeDescriptor SelectedObject
        {
            get { return this.SelectedItem as TypeDescriptor; }
        }

        public ObjectListBox(Type baseType)
            : base()
        {
            this.baseType = baseType;
            this.DrawMode = DrawMode.OwnerDrawFixed;
            this.ItemHeight = 70;
        }

        Type baseType;

        /// <summary>
        /// Method has to be called to give it functionality
        /// Cannot be called inside the constructor when using the Visual Studio designer...
        /// </summary>
        public virtual void Initialize()
        {
            this.DrawItem += new DrawItemEventHandler(ObjectListBox_DrawItem);
            
            this.Items.AddRange(ObjectsAvailable().ToArray());
            
            this.SelectedIndex = 0;
        }

        void ObjectListBox_DrawItem(object sender, DrawItemEventArgs e)
        {
            if (e.Index < 0 || e.Index > this.Items.Count)
                return;

            Rectangle name = e.Bounds;
            name.Height = 20;
            name.Offset(40, 0);
            name.Width -= 40;
            Rectangle properties = name;
            properties.Offset(0, name.Height);
            properties.Height = 20;
            Rectangle description = e.Bounds;
            description.Offset(0, name.Height + properties.Height);
            description.Height = 40;

            TypeDescriptor obj = this.Items[e.Index] as TypeDescriptor;
            e.DrawBackground();
            e.DrawFocusRectangle();
            lock (obj.Avatar)
            {
                e.Graphics.DrawImage(obj.Avatar, new Point(e.Bounds.X + 5, e.Bounds.Y + 5));
            }
            TextRenderer.DrawText(e.Graphics, obj.Name, new Font(Font, FontStyle.Bold),
                name, e.ForeColor, TextFormatFlags.Left | TextFormatFlags.VerticalCenter);

            TextRenderer.DrawText(e.Graphics, obj.Properties, Font,
                properties, Color.SteelBlue, TextFormatFlags.Left);

            TextRenderer.DrawText(e.Graphics, obj.Description, Font,
                description, Color.OrangeRed, TextFormatFlags.Left);
        }

        public IEnumerable<TypeDescriptor> ObjectsAvailable()
        {
            return baseType.EnumerateDerivedTypes().Select(t => TypeDescriptor.GetDescriptor(t));
        }
    }

    class TowerListBox : ObjectListBox
    {
        public TowerListBox() : base(typeof(TowerBase)) { }

        public TowerTypeDescriptor SelectedTower
        {
            get { return TowerTypeDescriptor.GetDescriptor(SelectedObject); }
        }

        public IEnumerable<TowerTypeDescriptor> TowersAvailable()
        {
            return typeof(TowerBase).EnumerateDerivedTypes().Select(t => TowerTypeDescriptor.GetDescriptor(t));
        }

        public override void Initialize()
        {
            base.Initialize();

            //sort towers with respect to the cost
            var temp = new List<TowerTypeDescriptor>();
            foreach (var item in this.Items)
            {
                temp.Add(TowerTypeDescriptor.GetDescriptor(item as TypeDescriptor));
            }
            this.Items.Clear();
            foreach (var item in temp.OrderBy(item => item.Cost))
            {
                this.Items.Add(item);
            }
            this.SelectedIndex = 0;
        }
    }

    class ShotListBox : ObjectListBox
    {
        public ShotListBox() : base(typeof(ShotBase)) { }

        public ShotTypeDescriptor SelectedShot
        {
            get { return ShotTypeDescriptor.GetDescriptor(SelectedObject); }
        }

        public IEnumerable<ShotTypeDescriptor> ShotsAvailable()
        {
            return typeof(ShotBase).EnumerateDerivedTypes().Select(t => ShotTypeDescriptor.GetDescriptor(t));
        }
    }

    class UnitListBox : ObjectListBox
    {
        public UnitListBox() : base(typeof(UnitBase)) { }

        public UnitTypeDescriptor SelectedUnit
        {
            get { return UnitTypeDescriptor.GetDescriptor(SelectedObject); }
        }

        public IEnumerable<UnitTypeDescriptor> UnitsAvailable()
        {
            return typeof(UnitBase).EnumerateDerivedTypes().Select(t => UnitTypeDescriptor.GetDescriptor(t));
        }
    }
}
