﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Drawing;
using SdlDotNet.Core;
using System.ComponentModel;
using System.Threading;

namespace TowerDefence
{
    class GameLogic
    {
        #region fields and properties

        List<TowerBase> towers;
        List<UnitBase> units;
        List<ShotBase> shots;
        IView view;
        Wave currentWave;
        BackgroundWorker backgroundWorker;
        public Player Player { get; private set; }
        WaveCollection waves;

        public enum GameState
        {
            Unstarted,
            Building,
            Waving,
            Finished,
        }

        public GameState State { get; private set; }

        public Size ViewSize { get { return view.Size; } }

        public Size MazeSize { get { return maze.GridDim; } }

        public string PlayerName { get { return Player.Name; } set { Player.Name = value; } }

        IMaze maze;

        private decimal mediPackCost = 5;
        public decimal MediPackCost { get { return mediPackCost; } private set { mediPackCost = value; } }
        public int MediPackHealth { get { return 1; } }

        private decimal towerBuildRightCost = 5;
        public decimal TowerBuildRightCost { get { return towerBuildRightCost; } private set { towerBuildRightCost = value; } }

        private int maxTowerNumber = 4;
        public int MaxTowerNumber { get { return maxTowerNumber; } private set { maxTowerNumber = value; } }

        #endregion

        #region events

        public event EventHandler<RefreshedEventArgs> Refreshed;

        public class WaveDefeatedEventArgs : EventArgs
        {
            public decimal PrizeMoney { get; private set; }
            public WaveDefeatedEventArgs(Wave wave)
            {
                PrizeMoney = wave.PrizeMoney;
            }
        }

        public event EventHandler<WaveDefeatedEventArgs> WaveDefeated;

        public event EventHandler<Wave.WaveCompletedEventArgs> WaveCompleted;

        public event EventHandler<Player.StateChangedEventArgs> PlayerDefeated;

        public event EventHandler<Player.StateChangedEventArgs> PlayerMoneyChanged;

        public event EventHandler<Player.StateChangedEventArgs> PlayerHealthChanged;

        public class WaveEventArgs : EventArgs
        {
            public Wave NewWave { get; private set; }
            public WaveEventArgs(Wave newWave)
            {
                NewWave = newWave;
            }
        }

        public event EventHandler<WaveEventArgs> NewWave;

        public event EventHandler<Wave.UnitSpawnedEventArgs> UnitSpawned;

        public class TowerNumberChangedEventArgs : EventArgs
        {
            public int? NumberOfTowers { get; private set; }
            public int? MaximumNumberOfTowers { get; private set; }
            public TowerNumberChangedEventArgs(int? NumberOfTowers, int? MaximumNumberOfTowers)
            {
                this.NumberOfTowers = NumberOfTowers;
                this.MaximumNumberOfTowers = MaximumNumberOfTowers;
            }
        }

        public event EventHandler<TowerNumberChangedEventArgs> TowerNumberChanged;

        #endregion

        #region ctor

        public GameLogic(int viewWidth = 300, int viewHeight = 200, int mazeWidth = 20, int mazeHeight = 11)
        {
            Sound.Enabled = true;

            maze = new Maze(new Size(mazeWidth, mazeHeight));

            view = new GDIView(new Size(viewWidth, viewHeight), MazeSize);
            //Events.TargetFps = 10;

            Player = new Player();
            Player.Defeated += new EventHandler<TowerDefence.Player.StateChangedEventArgs>(Player_Defeated);
            Player.PlayerMoneyChanged += new EventHandler<TowerDefence.Player.StateChangedEventArgs>(Player_PlayerMoneyChanged);
            Player.PlayerHealthChanged += new EventHandler<TowerDefence.Player.StateChangedEventArgs>(GameLogic_PlayerHealthChanged);
            towers = new List<TowerBase>();
            units = new List<UnitBase>();
            shots = new List<ShotBase>();
            view.Refreshed += new EventHandler<RefreshedEventArgs>(view_Refreshed);
            Events.Tick += new EventHandler<TickEventArgs>(Events_Tick);
            backgroundWorker = new BackgroundWorker();
            backgroundWorker.WorkerSupportsCancellation = true;
            backgroundWorker.WorkerReportsProgress = true;
            backgroundWorker.DoWork += new DoWorkEventHandler(backgroundWorker_DoWork);
            backgroundWorker.ProgressChanged += new ProgressChangedEventHandler(backgroundWorker_ProgressChanged);

            waves = new WaveCollection();
            //waves.AddTestWave<Rope>();
            //waves.AddTestWave<FireWizzrobe>();
            //waves.AddTestWave<DarkNut>();
            //waves.AddTestWave<DerivedUnit>();
            //waves.AddTestWave<Armos>();
            waves.AddStandardWaves();
            currentWave = TowerDefence.Wave.Empty;
        }

        #endregion

        #region manage towers, shots, and units

        private void AddTower(TowerBase tower)
        {
            tower.Shot += (sender, e) => AddShot(e.Shot);
            lock (towers)
            {
                towers.Add(tower);
            }
            view.Add(tower);
        }

        private void removeTower(TowerBase tower)
        {
            lock (towers)
            {
                towers.Remove(tower);
            }
            view.Remove(tower);
        }

        private void AddShot(ShotBase shot)
        {
            lock (shots)
            {
                shots.Add(shot);
            }
            view.Add(shot);
        }

        private void AddUnit(UnitBase unit)
        {
            unit.Accomplished += (sender, e) => Player.Hit(unit);
            lock (units)
            {
                units.Add(unit);
            }
            view.Add(unit);
        }

        /// <summary>
        /// Removes all objects which have deceased during the last tick event.
        /// Separated from the main tick event in order not to modify the lists while enumerating through them
        /// </summary>
        private void RemoveDeadObjects()
        {
            lock (towers)
            {
                foreach (var item in towers.Where(o => o.Dead).ToArray())
                {
                    towers.Remove(item);
                    view.Remove(item);
                }
            }
            lock (shots)
            {
                foreach (var item in shots.Where(o => o.Dead).ToArray())
                {
                    shots.Remove(item);
                    view.Remove(item);
                }
            }
            lock (units)
            {
                foreach (var item in units.Where(o => o.Dead).ToArray())
                {
                    units.Remove(item);
                    view.Remove(item);
                }
            }

            lock (currentWave)
            {
                if (currentWave.Dead && units.FirstOrDefault() == default(UnitBase))
                    EndWave();
            }
        }

        void EndWave()
        {
            Player.Money += currentWave.PrizeMoney;
            State = GameState.Building;

            lock (shots)
            {
                foreach (var item in shots.ToList())
                {
                    shots.Remove(item);
                    view.Remove(item);
                }
            }

            if (WaveDefeated != null)
                WaveDefeated(this, new WaveDefeatedEventArgs(currentWave));
        }

        #endregion

        #region event handlers

        void GameLogic_PlayerHealthChanged(object sender, Player.StateChangedEventArgs e)
        {
            if (PlayerHealthChanged != null)
                PlayerHealthChanged(this, e);
        }

        void Player_PlayerMoneyChanged(object sender, Player.StateChangedEventArgs e)
        {
            if (PlayerMoneyChanged != null)
                PlayerMoneyChanged(this, e);
        }

        void Player_Defeated(object sender, Player.StateChangedEventArgs e)
        {
            if (PlayerDefeated != null)
                PlayerDefeated(this, e);
            Finish();
        }

        void currentWave_UnitSpawned(object sender, Wave.UnitSpawnedEventArgs e)
        {
            if (UnitSpawned != null)
                UnitSpawned(this, e);
        }

        void currentWave_WaveCompleted(object sender, Wave.WaveCompletedEventArgs e)
        {
            if (WaveCompleted != null)
                WaveCompleted(this, e);
        }

        //wrap the IView.Refreshed event via the BackgroundWorker.ProgressChanged event
        //into the GameLogic.Refreshed event to access the main thread
        void view_Refreshed(object sender, RefreshedEventArgs e)
        {
            backgroundWorker.ReportProgress(0, e); 
        }
        void backgroundWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            if (Refreshed != null)
                Refreshed(this, e.UserState as RefreshedEventArgs);
        }

        Action CancelBackgroundWorker;//Closure to capture e.Cancel for cancelling the backgroundworker
        void backgroundWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            Thread.CurrentThread.IsBackground = true;
            CancelBackgroundWorker = () => e.Cancel = true;
            Events.Run();
        }

        /// <summary>
        /// The SDL Events_Tick event handler as timer thread for the game physics
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void Events_Tick(object sender, TickEventArgs e)
        {
            if (State == GameState.Waving)
            {
                IEnumerable<BaseObject> temp;//keep locked time short
                lock (towers)
                {
                    temp = towers.ToList();
                }
                foreach (var item in temp)
                    item.Update(e);
                lock (shots)
                {
                    temp = shots.ToList();
                }
                foreach (var item in temp)
                    item.Update(e);
                lock (units)
                {
                    temp = units.ToList();
                }
                foreach (var item in temp)
                    item.Update(e);

                RemoveDeadObjects();

                lock (currentWave)
                {
                    foreach (var item in currentWave.Update(e))
                    {
                        item.Path = new Path(maze.GetPath(item.Strategy));
                        AddUnit(item);
                    }
                }
            }

            if (backgroundWorker.CancellationPending)
            {
                view.Dispose();
                Events.QuitApplication();
                CancelBackgroundWorker();
            }
        }

        #endregion

        #region exposed functions

        /// <summary>
        /// Starts the SDL event loop and therefore the game
        /// </summary>
        public void Start()
        {
            if (State == GameState.Unstarted)
            {
                Player.HitPoints = 10;
                Player.Money = 3;
                if (TowerNumberChanged != null)
                    TowerNumberChanged(this, new TowerNumberChangedEventArgs(0, MaxTowerNumber));
                view.Start();
                backgroundWorker.RunWorkerAsync();
                State = GameState.Building;
            }
        }

        /// <summary>
        /// Ends build phase and starts next Wave
        /// </summary>
        public void Wave()
        {
            if (State == GameState.Building)
            {
                State = GameState.Waving;

                lock (currentWave)
                {
                    currentWave = waves.NextWave();
                    currentWave.WaveCompleted += new EventHandler<TowerDefence.Wave.WaveCompletedEventArgs>(currentWave_WaveCompleted);
                    currentWave.UnitSpawned += new EventHandler<TowerDefence.Wave.UnitSpawnedEventArgs>(currentWave_UnitSpawned);

                    if (NewWave != null)
                        NewWave(this, new WaveEventArgs(currentWave));
                }
            }
        }

        /// <summary>
        /// Ends and disposes the game
        /// </summary>
        public void Finish()
        {
            State = GameState.Finished;
            backgroundWorker.CancelAsync();
            view.Dispose();
        }

        /// <summary>
        /// try to build a tower
        /// </summary>
        /// <param name="point">point to build the tower on</param>
        /// <param name="TowerType">type of the tower to be buildt</param>
        internal bool BuildTower(Point point, TowerTypeDescriptor TowerType)
        {
            bool isRock = TowerType.Type == typeof(Derived.Rock);
            int numberOfShootingTowers = towers.Count(tb => tb.GetType() != typeof(Derived.Rock));
            if (State == GameState.Building && TowerType.Cost <= Player.Money
                && (isRock || numberOfShootingTowers < MaxTowerNumber))
            {
                TowerBase newTower = System.Reflection.Assembly.GetExecutingAssembly().CreateInstance(TowerType.Type.FullName) as TowerBase;
                Point mazeCoordinates = view.TransformScreenToMaze(point).ToPoint();
                newTower.MazeCoordinates = mazeCoordinates;
                newTower.Targets = units;
                if (maze.TryBuild(mazeCoordinates))
                {
                    Player.Money -= TowerType.Cost;
                    AddTower(newTower);
                    if (!isRock && TowerNumberChanged != null)
                    {
                        TowerNumberChanged(this, new TowerNumberChangedEventArgs(numberOfShootingTowers + 1, null));
                    }
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// try to remove a tower
        /// </summary>
        /// <param name="point">point to remove the tower from</param>
        internal void RemoveTower(Point point)
        {
            if (State == GameState.Building)
            {
                var mazePosition = view.TransformScreenToMaze(point).ToPoint();
                var towerToRemove = towers.Where(t => t.MazeCoordinates.ToPoint() == mazePosition).FirstOrDefault();
                if (towerToRemove != null)
                {
                    Player.Money += towerToRemove.Cost / 2;
                    removeTower(towerToRemove);
                    maze.TryRemove(mazePosition);
                    if (towerToRemove.GetType() != typeof(Derived.Rock) && TowerNumberChanged != null)
                    {
                        TowerNumberChanged(this, new TowerNumberChangedEventArgs(towers.Count(tb => tb.GetType() != typeof(Derived.Rock)), null));
                    }
                }
            }
        }

        internal void SetSoundSettings(Settings.SoundSettingsChangedEventArgs e)
        {
            Sound.Enabled = e.Enabled;
            Sound.Volume = e.Volume;
        }

        internal bool BuyMediPack()
        {
            if (Player.Money > MediPackCost)
            {
                Player.HitPoints += MediPackHealth;
                var cost = MediPackCost;
                MediPackCost *= 2;
                Player.Money -= cost;
                return true;
            }
            else
                return false;
        }

        internal bool BuyTowerBuildRight()
        {
            if (Player.Money > TowerBuildRightCost)
            {
                MaxTowerNumber++;
                if (TowerNumberChanged != null)
                {
                    TowerNumberChanged(this, new TowerNumberChangedEventArgs(null, MaxTowerNumber));
                }
                var cost = TowerBuildRightCost;
                TowerBuildRightCost *= 2;
                Player.Money -= cost;
                return true;
            }
            else
                return false;
        }

        #endregion
    }
}
