﻿using System;
using System.Drawing;
using System.Reflection;
using System.Linq;
using System.Text;
using System.Collections.Generic;

namespace TowerDefence
{
    /// <summary>
    /// identifies properties to be listed in the TypeDescriptor
    /// </summary>
    [AttributeUsage(AttributeTargets.Property)]
    class DescriptorPropertyAttribute : Attribute
    {
    }

    /// <summary>
    /// classes to obtain information about all available objects in the game 
    /// and to supply the UI with them via reflection for flexibility
    /// </summary>
    public class TypeDescriptor
    {
        public string Name { get; private set; }

        public Bitmap Avatar { get; private set; }

        public string Description { get; private set; }

        public string Properties { get; private set; }

        public Type Type { get; private set; }

        public float Speed { get; private set; }

        public double Rating { get; private set; }

        protected BaseObject exampleInstance;

        protected TypeDescriptor(Type baseType)
        {
            exampleInstance = Assembly.GetExecutingAssembly().CreateInstance(baseType.FullName) as BaseObject;
            Avatar = exampleInstance.Sprite.CurrentFrame;
            Description = exampleInstance.Description;
            Type = baseType;
            Name = exampleInstance.NickName == string.Empty ? baseType.Name : exampleInstance.NickName;

            var descriptorProperties = Type.GetProperties().Where(pInfo => pInfo.GetCustomAttributes(typeof(DescriptorPropertyAttribute), true).Length > 0).OrderBy(pInfo=>pInfo.Name);
            if (descriptorProperties.FirstOrDefault() == null)
                Properties = String.Empty;
            else
            {
                var propertyValues = descriptorProperties.Select(dP => dP.Name + ": " + dP.GetValue((object)exampleInstance, null)).ToArray();

                StringBuilder sb = new StringBuilder();
                sb.Append(propertyValues[0]);

                for (int i = 1; i < propertyValues.Length; i++)
                {
                    sb.Append(", ");
                    sb.Append(propertyValues[i]);
                }
                Properties = sb.ToString();
            }

            Speed = exampleInstance.Speed;
            Rating = exampleInstance.Rating;
        }

        public override string ToString()
        {
            return Name;
        }

        //Work as singletons
        static protected Dictionary<Type, TypeDescriptor> descriptors = new Dictionary<Type, TypeDescriptor>();

        static public TypeDescriptor GetDescriptor(Type baseType)
        {
            if (!baseType.IsSubclassOf(typeof(BaseObject)))
                throw new TypeDescriptorException(baseType.FullName + " is not derived from " + typeof(BaseObject).FullName);
            if (!descriptors.ContainsKey(baseType))
                descriptors.Add(baseType, new TypeDescriptor(baseType));
            return descriptors[baseType];
        }
    }

    public class TowerTypeDescriptor : TypeDescriptor
    {
        public float Range { get; private set; }

        public ShotTypeDescriptor ShotType { get; private set; }

        public decimal Cost { get; private set; }

        protected TowerTypeDescriptor(Type baseType)
            : base(baseType)
        {
            Range = (exampleInstance as TowerBase).Range;
            ShotType = (exampleInstance as TowerBase).ShotType;
            Cost = (exampleInstance as TowerBase).Cost;
        }

        //Work as singletons
        static new public TowerTypeDescriptor GetDescriptor(Type baseType)
        {
            if (!baseType.IsSubclassOf(typeof(TowerBase)))
                throw new TypeDescriptorException(baseType.FullName + " is not derived from " + typeof(TowerBase).FullName);
            if (!descriptors.ContainsKey(baseType))
                descriptors.Add(baseType, new TowerTypeDescriptor(baseType));
            return descriptors[baseType] as TowerTypeDescriptor;
        }

        static new public TowerTypeDescriptor GetDescriptor(TypeDescriptor baseTypeDescriptor)
        {
            if (!baseTypeDescriptor.Type.IsSubclassOf(typeof(TowerBase)))
                throw new TypeDescriptorException(baseTypeDescriptor.Type.FullName + " is not derived from " + typeof(TowerBase).FullName);
            if (baseTypeDescriptor.GetType() != typeof(TowerTypeDescriptor))
            {
                descriptors.Remove(baseTypeDescriptor.Type);
                descriptors.Add(baseTypeDescriptor.Type, new TowerTypeDescriptor(baseTypeDescriptor.Type));
            }
            return descriptors[baseTypeDescriptor.Type] as TowerTypeDescriptor;
        }
    }

    public class ShotTypeDescriptor : TypeDescriptor
    {
        public float Damage { get; private set; }

        public float ImpactRadius { get; private set; }

        protected ShotTypeDescriptor(Type baseType)
            : base(baseType)
        {
            Damage = (exampleInstance as ShotBase).Damage;
            ImpactRadius = (exampleInstance as ShotBase).ImpactRadius;
        }

        //Work as singletons
        static new public ShotTypeDescriptor GetDescriptor(Type baseType)
        {
            if (!baseType.IsSubclassOf(typeof(ShotBase)))
                throw new TypeDescriptorException(baseType.FullName + " is not derived from " + typeof(ShotBase).FullName);
            if (!descriptors.ContainsKey(baseType))
                descriptors.Add(baseType, new ShotTypeDescriptor(baseType));
            return descriptors[baseType] as ShotTypeDescriptor;
        }

        static new public ShotTypeDescriptor GetDescriptor(TypeDescriptor baseTypeDescriptor)
        {
            if (!baseTypeDescriptor.Type.IsSubclassOf(typeof(ShotBase)))
                throw new TypeDescriptorException(baseTypeDescriptor.Type.FullName + " is not derived from " + typeof(ShotBase).FullName);
            if (baseTypeDescriptor.GetType() != typeof(ShotTypeDescriptor))
            {
                descriptors.Remove(baseTypeDescriptor.Type);
                descriptors.Add(baseTypeDescriptor.Type, new ShotTypeDescriptor(baseTypeDescriptor.Type));
            }
            return descriptors[baseTypeDescriptor.Type] as ShotTypeDescriptor;
        }
    }

    public class UnitTypeDescriptor : TypeDescriptor
    {
        public MazeStrategy Strategy { get; private set; }

        public int HitPoints { get; private set; }

        public int Damage { get; private set; }

        protected UnitTypeDescriptor(Type baseType)
            : base(baseType)
        {
            Strategy = (exampleInstance as UnitBase).Strategy;
            HitPoints = (exampleInstance as UnitBase).HitPoints;
            Damage = (exampleInstance as UnitBase).Damage;
        }

        //Work as singletons
        static new public UnitTypeDescriptor GetDescriptor(Type baseType)
        {
            if (!baseType.IsSubclassOf(typeof(UnitBase)))
                throw new TypeDescriptorException(baseType.FullName + " is not derived from " + typeof(UnitBase).FullName);
            if (!descriptors.ContainsKey(baseType))
                descriptors.Add(baseType, new UnitTypeDescriptor(baseType));
            return descriptors[baseType] as UnitTypeDescriptor;
        }

        static new public UnitTypeDescriptor GetDescriptor(TypeDescriptor baseTypeDescriptor)
        {
            if (!baseTypeDescriptor.Type.IsSubclassOf(typeof(UnitBase)))
                throw new TypeDescriptorException(baseTypeDescriptor.Type.FullName + " is not derived from " + typeof(UnitBase).FullName);
            if (baseTypeDescriptor.GetType() != typeof(UnitTypeDescriptor))
            {
                descriptors.Remove(baseTypeDescriptor.Type);
                descriptors.Add(baseTypeDescriptor.Type, new UnitTypeDescriptor(baseTypeDescriptor.Type));
            }
            return descriptors[baseTypeDescriptor.Type] as UnitTypeDescriptor;
        }
    }

    public class TypeDescriptorException : Exception
    {
        public TypeDescriptorException(string message) : base(message) { }
    }
}