﻿//#define StageThread

using System;
using System.Collections.Generic;
using System.Linq;
using System.Drawing;
#if StageThread
using System.Threading;
#else
using System.Timers;
#endif

namespace Animation
{
    /// <summary>
    /// draws a set of animated sprites to a bitmap
    /// </summary>
    class Stage : IDisposable
    {
        List<Sprite> sprites;
        public Bitmap Image { get; private set; }
        private Bitmap background;
        public Bitmap Background
        {
            get { return background; }
            set { background = new Bitmap(value, Image.Size); }
        }
        Graphics g;
        Size size;
        public Size Size
        {
            get { return size; }
        }
        Timer timer;
        public int Period { get; private set; }

        public event EventHandler UpdatedEvent;

        public Stage(Size size, int period)
        {
            this.size = size;
            Image = new Bitmap(size.Width, size.Height);
            g = Graphics.FromImage(Image);
            sprites = new List<Sprite>();
            Period = period;
        }

        public void Start()
        {
#if StageThread
            timer = new Timer(new TimerCallback(Update), null, 0, Period);
#else
            timer = new Timer(Period); timer.Elapsed += (sender, e) => { Update(null); }; timer.Start();
#endif
        }

        public void Stop()
        {
            timer.Dispose();
        }

        void Update(object o)
        {
            List<Sprite> temp;
            lock (sprites)
            {
                temp = sprites.ToList();
            }
            temp.Sort(new Comparison<Sprite>((s1, s2) => s1.Z - s2.Z));
            lock (g)
            {
                if (background != null)
                {
                    g.DrawImage(background, new Point(0, 0));
                }
                foreach (var item in temp)
                {
                    item.Update(g);
                }
            }
            if (UpdatedEvent != null)
            {
                UpdatedEvent(this, EventArgs.Empty);
            }
        }

        public void Add(Sprite s)
        {
            lock (sprites)
            {
                sprites.Add(s);
            }
        }

        public void Remove(Sprite s)
        {
            lock (sprites)
            {
                sprites.Remove(s);
            }
        }

        public void Dispose()
        {
            if (timer != null)
                timer.Dispose();
            if (g != null)
                g.Dispose();
        }
    }
}
