﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using SdlDotNet.Core;
using System.Diagnostics;

namespace TowerDefence
{
    public class ShotBase : BaseObject
    {
        private PointF directionOfMotion;

        public PointF DirectionOfMotion
        {
            get { return directionOfMotion; }
            set { directionOfMotion = value.Normalize(); }
        }

        [DescriptorProperty]
        public virtual int Damage { get { return 1; } }

        public virtual float ImpactRadius { get { return 0.5f; } }

        public List<UnitBase> Targets { get; set; }

        /// <summary>
        /// time in ms until the shot can hit an object and can thus be destroyed
        /// </summary>
        protected int TimeToDamagability { get; set; }

        public class HitEventArgs : EventArgs
        {
            public UnitBase HittedUnit { get; private set; }
            public HitEventArgs(UnitBase hittedUnit)
            {
                HittedUnit = hittedUnit;
            }
        }

        public event EventHandler<HitEventArgs> Hit;

        /// <summary>
        /// Constructs ShotBase object
        /// </summary>
        /// <param name="position">in maze coordinates</param>
        /// <param name="directionOfMotion">in maze coordinates</param>
        /// <param name="Targets">reference to a list with possible targets</param>
        public ShotBase()
        {
            TimeToDamagability = (int)(200 / Speed);

            var newFireSound = this.GetType().getNewField("FireSound");
            if (newFireSound.GetValue(this) == null)
                newFireSound.SetValue(this, new Sound(FireSoundFile));
            (newFireSound.GetValue(this) as Sound).Play();
        }

        public override void Update(TickEventArgs args)
        {
            MazeCoordinates = MazeCoordinates.Add(DirectionOfMotion.Scale((Speed / 1000) * args.TicksElapsed));

            base.Update(args);

            Sprite.Z += 100;

            if (TimeToDamagability > 0)
            {
                TimeToDamagability -= args.TicksElapsed;
            }
            else
            {
                var nearestTarget = Targets.OrderBy(u => DistanceTo(u.MazeCoordinates)).FirstOrDefault();
                if (nearestTarget != null && DistanceTo(nearestTarget.MazeCoordinates) < ImpactRadius)
                {
                    nearestTarget.Hit(this);
                    Dead = true;
                    if (Hit != null)
                        Hit(this, new HitEventArgs(nearestTarget));
                }
            }
        }

        protected static Sound FireSound;

        protected virtual string FireSoundFile { get { return "Wowpulse.wav"; } }

        protected override string SpriteFile { get { return "sunshot.bmp"; } }

        protected override int[] FramesPerDirection { get { return new int[] { 20 }; } }

        protected override int AnimationDelay { get { return 15; } }

        [DescriptorProperty]
        public override float Speed { get { return 10f; } }

        public override string Description { get { return @"base of all shots - low damage potential and speed"; } }

        protected new static List<Animation.Animation> animations;


        public override double Rating
        {
            get { return Math.Log(Speed + 1) * Damage * Math.Log(ImpactRadius + 1); }
        }
    }
}
