/*
 * -------------------------------------------
 * PathFinderNode STRUCT
 * -------------------------------------------
 */
var PathFinderNode = Class.extend({
    init: function() {
        this.F = 0;
        this.G = 0;
        this.H = 0;
        this.X = 0;
        this.Y = 0;
        this.PX = 0;
        this.PY = 0;
    },
    clone: function() {
        var me = this;
        var you = new PathFinderNode();
        you.F = me.F;
        you.G = me.G;
        you.H = me.H;
        you.X = me.X;
        you.Y = me.Y;
        you.PX = me.PX;
        you.PY = me.PY;
        return you;
    }
});

/*
 * -------------------------------------------
 * PriorityQueueB CLASS
 * -------------------------------------------
 */
var PriorityQueueB = Class.extend({
    init: function(comparer, type) {
        this.type = type;
        this.innerList = [];
        this.comparer = comparer;
    },
    switchElements: function(i, j) {
        var h = this.innerList[i];
        this.innerList[i] = this.innerList[j];
        this.innerList[j] = h;
    },
    onCompare: function(i, j) {
        return this.comparer(this.innerList[i], this.innerList[j]);
    },
    push: function(item) {
        var p = this.innerList.length, p2;
        this.innerList.push(item);

        do {
            if(p === 0)
                break;

            p2 = parseInt((p - 1) / 2);

            if(this.onCompare(p, p2) < 0) {
                this.switchElements(p, p2);
                p = p2;
            } else
                break;
        } while(true);

        return p;
    },
    pop: function() {
        var result = this.innerList[0];
        var p = 0, p1, p2, pn;
        this.innerList[0] = this.innerList[this.innerList.length - 1];
        this.innerList.splice(this.innerList.length - 1, 1);

        do {
            pn = p;
            p1 = 2 * p + 1;
            p2 = 2 * p + 2;

            if(this.innerList.length > p1 && this.onCompare(p, p1) > 0)
                p = p1;

            if(this.innerList.length > p2 && this.onCompare(p, p2) > 0)
                p = p2;
            
            if (p === pn)
                break;

            this.switchElements(p, pn);
        } while(true);

        return result;
    },
    update: function(i) {
        var p = i,pn;
        var p1, p2;

        do {
            if(p === 0)
                break;

            p2 = parseInt((p - 1) / 2);

            if(this.onCompare(p, p2) < 0) {
                this.switchElements(p, p2);
                p = p2;
            } else
                break;
        } while(true);

        if(p < i)
            return;

        do {
            pn = p;
            p1 = 2 * p + 1;
            p2 = 2 * p + 2;

            if(this.innerList.length > p1 && this.onCompare(p, p1) > 0)
                p = p1;

            if(this.innerList.length > p2 && this.oCompare(p, p2) > 0)
                p = p2;
            
            if(p === pn)
                break;

            this.switchElements(p, pn);
        } while(true);
    },
    peek: function() {
        if(this.innerList.length)
            return this.innerList[0];

        return new this.type();
    },
    clear: function() {
        this.innerList = [];
    },
    count: function() {
        return this.innerList.length;
    },
    removeLocation: function(item) {
        var index = -1;

        for(var i = 0; i < this.innerList.length; i++) {
            if (this.comparer.compare(this.innerList[i], item) === 0)
                index = i;
        }

        if (index !== -1)
            this.innerList.splice(index, 1);
    },
    get: function(index) {
        return this.innerList[index];
    },
    set: function(index, value) { 
        this.innerList[index] = value;
        this.update(index);
    }
});

/*
 * -------------------------------------------
 * Point STRUCT
 * -------------------------------------------
 */
var Point = Class.extend({
    init: function(x, y) {
        this.X = x;
        this.Y = y;
    },
    clone: function() {
        var me = this;
        var you = new Point(me.X, me.Y);
        return you;
    }
});

/*
 * -------------------------------------------
 * Size STRUCT
 * -------------------------------------------
 */
var Size = Class.extend({
    init: function(width, height) {
        this.Width = width;
        this.Height = height;
    },
    clone: function() {
        var me = this;
        var you = new Size(me.Width, me.Height);
        return you;
    }
});

/*
 * -------------------------------------------
 * PathFinderNodeType ENUM
 * -------------------------------------------
 */
var PathFinderNodeType = {
    Start   : 1,
    End     : 2,
    Open    : 4,
    Close   : 8,
    Current : 16,
    Path    : 32
};

/*
 * -------------------------------------------
 * MazeStrategy ENUM
 * -------------------------------------------
 */
var MazeStrategy = {
    Manhattan           : 1,
    MaxDXDY             : 2,
    DiagonalShortCut    : 3,
    Euclidean           : 4,
    EuclideanNoSQR      : 5,
    Custom1             : 6,
    Air                 : 7
};

/*
 * -------------------------------------------
 * Maze CLASS
 * -------------------------------------------
 */
var Maze = Class.extend({
    init: function(size, start, end) {
        this.gridDim = size;
        this.grid = [];

        for (var i = 0; i < size.Width + 1; i++) {
            this.grid.push([]);

            for (var j = 0; j < size.Height + 1; j++)
                this.grid[i].push(1);
        }

        this.start = start || new Point(0, parseInt(size.Height / 2));
        this.end = end || new Point(size.Width, parseInt(size.Height / 2));
        this.pf = new PathFinder(this.grid);
        this.paths = {};
    },
    isPointInGrid: function(point) {
        if (point.X > this.gridDim.Width - 1 || point.X < 0)
            return false;

        if (point.Y > this.gridDim.Height - 1 || point.Y < 0)
            return false;

        return true;
    },
    tryBuild: function(point) {
        if (this.grid[point.X][point.Y] === 0)
            return false;

        if (point.X === this.gridDim.Width - 1 || point.X === 0)
            return false;

        this.grid[point.X][point.Y] = 0;
        this.pf.formula = MazeStrategy.Euclidean;
        this.pf.diagonals = false;

        if (this.pf.findPath(this.start, this.end)) {
            this.paths = {};
            return true;
        } else {
            this.grid[point.X][point.Y] = 1;
            return false;
        }
    },
    tryRemove: function(point) {
        if (this.grid[point.X][point.Y] !== 1) {
            this.grid[point.X][point.Y] = 1;
            this.paths = {};
            return true;
        }
        
        return false;
    },
    getPath: function(mazeStrategy) {
        var path = [];

        if(mazeStrategy === MazeStrategy.Air)
            return getPathAir();

        if (!this.paths[mazeStrategy])
            this.calculate(mazeStrategy);
        
        return this.paths[mazeStrategy];
    },
    getPathAir: function() {
        var path = [];

        for (var i = this.start.X; i < this.end.X + 1; i++)
            path.push(new Point(i, this.start.Y));

        return path;
    },
    calculate: function(mazeStrategy) {
        var path;

        if (mazeStrategy === MazeStrategy.Air) {
            path = this.getPathAir();
        } else {
            this.pf.formula = mazeStrategy;
            this.pf.diagonals = false;
            var n = this.pf.findPath(this.start, this.end);
            path = [];

            for(var i = 0; i < n.length; i++)
                path.push(new Point(n[i].X, n[i].Y));
        }

        this.paths[mazeStrategy] = path;
    }
});

/*
 * -------------------------------------------
 * PathFinder CLASS
 * -------------------------------------------
 */
var PathFinder = Class.extend({
    init: function(grid) {
        this.grid = grid;
        this.formula = MazeStrategy.Manhattan;
        this.horiz = 0;
        this.stopped = true;
        this.diagonals = true;
        this.stop = false;
        this.heuristicEstimate = 2;
        this.punishChangeDirection = false;
        this.reopenCloseNodes = false;
        this.tieBreaker = false;
        this.heavyDiagonals = false;
        this.searchLimit = 2000;
        this.reset();  
    },
    reset: function() {
        this.close = [];
        this.open = new PriorityQueueB(function(x, y) {
            if (x.F > y.F)
                return 1;
            else if (x.F < y.F)
                return -1;

            return 0;
        }, PathFinderNode);
    },
    findPath: function(start, end) {
        var me = this;
        var parentNode = new PathFinderNode();
        var found = false;
        var gridX = me.grid.length;
        var gridY = me.grid[0].length;
        me.stop = false;
        me.stopped = false;
        me.reset();
        var direction = me.diagonals ? [[0, -1], [1, 0], [0, 1], [-1, 0], [1, -1], [1, 1], [-1, 1], [-1, -1]] : [[0, -1], [1, 0], [0, 1], [-1, 0]];
	var ndiag = me.diagonals ? 8 : 4;

        parentNode.G = 0;
        parentNode.H = me.heuristicEstimate;
        parentNode.F = parentNode.G + parentNode.H;
        parentNode.X = start.X;
        parentNode.Y = start.Y;
        parentNode.PX = parentNode.X;
        parentNode.PY = parentNode.Y;
        me.open.push(parentNode);

        while(me.open.count() > 0 && !me.stop) {
            parentNode = me.open.pop();

            if (parentNode.X === end.X && parentNode.Y === end.Y) {
                me.close.push(parentNode);
                found = true;
                break;
            }

            if (me.close.length > me.searchLimit) {
                me.stopped = true;
                return;
            }

            if (me.punishChangeDirection)
                me.horiz = (parentNode.X - parentNode.PX); 

            for (var i = 0; i < ndiag; i++) {
                var newNode = new PathFinderNode();
                newNode.X = parentNode.X + direction[i][0];
                newNode.Y = parentNode.Y + direction[i][1];

                if (newNode.X < 0 || newNode.Y < 0 || newNode.X >= gridX || newNode.Y >= gridY)
                    continue;

                var newG = 0;

                if (me.heavyDiagonals && i > 3)
                    newG = parentNode.G + parseInt(me.grid[newNode.X][newNode.Y] * 2.41);
                else
                    newG = parentNode.G + this.grid[newNode.X][newNode.Y];

                if (newG === parentNode.G)
                    continue;

                if (me.punishChangeDirection) {
                    if (newNode.X - parentNode.X) {
                        if (!me.horiz)
                            newG += 20;
                    }

                    if (newNode.Y - parentNode.Y) {
                        if (me.horiz)
                            newG += 20;
                    }
                }

                var foundInOpenIndex = -1;

                for(var j = 0, n = me.open.count(); j < n; j++) {
                    if (me.open.get(j).X === newNode.X && me.open.get(j).Y === newNode.Y) {
                        foundInOpenIndex = j;
                        break;
                    }
                }

                if (foundInOpenIndex !== -1 && me.open.get(foundInOpenIndex).G <= newG)
                    continue;

                var foundInCloseIndex = -1;

                for(var j = 0, n = me.close.length; j < n; j++) {
                    if (me.close[j].X === newNode.X && me.close[j].Y === newNode.Y) {
                        foundInCloseIndex = j;
                        break;
                    }
                }

                if (foundInCloseIndex !== -1 && (me.reopenCloseNodes || me.close[foundInCloseIndex].G <= newG))
                    continue;

                newNode.PX = parentNode.X;
                newNode.PY = parentNode.Y;
                newNode.G = newG;

                switch(me.formula) {
                    case MazeStrategy.MaxDXDY:
                        newNode.H = me.heuristicEstimate * (Math.max(Math.abs(newNode.X - end.X), Math.abs(newNode.Y - end.Y)));
                        break;

                    case MazeStrategy.DiagonalShortCut:
                        var h_diagonal = Math.min(Math.abs(newNode.X - end.X), Math.abs(newNode.Y - end.Y));
                        var h_straight = (Math.abs(newNode.X - end.X) + Math.abs(newNode.Y - end.Y));
                        newNode.H = (me.heuristicEstimate * 2) * h_diagonal + me.heuristicEstimate * (h_straight - 2 * h_diagonal);
                        break;

                    case MazeStrategy.Euclidean:
                        newNode.H = parseInt(me.heuristicEstimate * Math.sqrt(Math.pow((newNode.X - end.X) , 2) + Math.pow((newNode.Y - end.Y), 2)));
                        break;

                    case MazeStrategy.EuclideanNoSQR:
                        newNode.H = parseInt(me.heuristicEstimate * (Math.pow((newNode.X - end.X) , 2) + Math.pow((newNode.Y - end.Y), 2)));
                        break;

                    case MazeStrategy.Custom1:
                        var dxy = new Point(Math.abs(end.X - newNode.X), Math.abs(end.Y - newNode.Y));
                        var Orthogonal = Math.abs(dxy.X - dxy.Y);
                        var Diagonal = Math.abs(((dxy.X + dxy.Y) - Orthogonal) / 2);
                        newNode.H = me.heuristicEstimate * (Diagonal + Orthogonal + dxy.X + dxy.Y);
                        break;
						
                    case MazeStrategy.Manhattan:
                    default:
                        newNode.H = me.heuristicEstimate * (Math.abs(newNode.X - end.X) + Math.abs(newNode.Y - end.Y));
                        break;
                }

                if (me.tieBreaker) {
                    var dx1 = parentNode.X - end.X;
                    var dy1 = parentNode.Y - end.Y;
                    var dx2 = start.X - end.X;
                    var dy2 = start.Y - end.Y;
                    var cross = Math.abs(dx1 * dy2 - dx2 * dy1);
                    newNode.H = parseInt(newNode.H + cross * 0.001);
                }

                newNode.F = newNode.G + newNode.H;
                me.open.push(newNode);
            }

            me.close.push(parentNode);
        }

        if (found) {
            var fNode = me.close[me.close.length - 1].clone();

            for(var i = me.close.length - 1; i >= 0; i--) {
                if ((fNode.PX === me.close[i].X && fNode.PY === me.close[i].Y) || i === me.close.Count - 1)
                    fNode = me.close[i].clone();
                else
                    me.close.splice(i, 1);
            }

            me.stopped = true;
            return me.close;
        }

        me.stopped = true;
    }
});
