﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using SdlDotNet;
using SdlDotNet.Graphics;
using SdlDotNet.Core;
using System.Drawing;
using SdlDotNet.Input;
using System.Windows.Forms;
using ProtoBuf;
using GameLibrary;
using System.IO;


namespace GameClient
{
    class Screen
    {
        const int SCREEN_WIDTH = 800;
        const int SCREEN_HEIGHT = 600;
        Surface screen;
        Surface buffer;
        Dictionary<string,Painter> painters;
        MapPainter mappainter;
        PlayerPainter playerpainter;
        Client client;
        Map map;
        input currentinput = input.nonnullbyte;

        public Surface Buffer
        {
            get { return buffer; }
            set { buffer = value; }
        }
        public int Width
        {
            get { return SCREEN_WIDTH; }
        }
        public int Height
        {
            get { return SCREEN_HEIGHT; }
        }
        public Dictionary<string, Spritesheet> Spritesheets
        {
            get;
            set;
        }

        public Screen()
        {
            screen = Video.SetVideoMode(SCREEN_WIDTH, SCREEN_HEIGHT);
            buffer = new Surface(screen);
            screen.Fill(Color.Chocolate);
            screen.Update();
            buffer.Update();
            loadSpritesheets();
            ConnectToServer();
            painters = new Dictionary<string,Painter>();
            Events.Quit += Close;
            Events.Tick += Tick;
            Events.KeyboardDown += KeyDownUp;
            Events.KeyboardUp += KeyDownUp;
            Events.MouseMotion += MouseMotion;
            Events.Run();
        }

        private void MouseMotion(object sender, SdlDotNet.Input.MouseMotionEventArgs e)
        {
            if (mappainter == null) return;
            if (e.ButtonPressed)
            {
                var p = mappainter;
                
                p.Center = new Point(p.Center.X - e.RelativeX, p.Center.Y - e.RelativeY);
            }
        }
        [Flags]
        enum input
        {
            forward = 0x1,
            backward =0x2,
            turnLeft = 0x4,
            turnRight = 0x8,
            nonnullbyte = 0xF000
        }

        private void KeyDownUp(object sender, SdlDotNet.Input.KeyboardEventArgs e)
        {
            input toggle = 0;
            if (mappainter == null) return;
            switch (e.Key)
            {
                case Key.W:
                    toggle = input.forward;
                    break;
                case Key.A:
                    toggle = input.turnLeft;
                    break;
                case Key.S:
                    toggle = input.backward;
                    break;
                case Key.D:
                    toggle = input.turnRight;
                    break;
            }
            if (e.Down) currentinput = currentinput | toggle;
            else currentinput = currentinput & ~toggle;
            client.Send(BitConverter.GetBytes((int)currentinput));
            
        }
        public void loadSpritesheets()
        {
            Spritesheets = new Dictionary<string, Spritesheet>();
            Spritesheets.Add("Map", new Spritesheet(Images.MapTiles, 40, screen));
            Spritesheets.Add("Player", new Spritesheet(Images.Player,70, screen,true));
        }
        public void ConnectToServer()
        {
            client = new Client("127.0.0.1", 8080);
           
            client.Received += (sender, data) => {
                var frame = Serializer.DeserializeWithLengthPrefix<Frame>(new MemoryStream(data), PrefixStyle.Fixed32);
                if (frame.Map != null)
                {
                    map = frame.Map;
                    mappainter = new MapPainter(this, map);
                    painters.Add("mappainter",mappainter);
                }
                UpdatePlayer(frame.Player, client.ID);
                if (frame.Entities != null)
                {
                    foreach (var entity in frame.Entities)
                    {
                        if (painters.ContainsKey(entity.Id))
                        {
                            var painter = painters[entity.Id] as EntityPainter;
                            painter.Update(entity, playerpainter.Position);
                        }
                        else
                        {
                            painters[entity.Id] = new EntityPainter(this, entity, playerpainter.Position);
                        }
                    }
                }
                
                
            };
            client.Disconnected += sender => {
                MessageBox.Show("Disconnected from Server");
                Close(sender, null); 
            };
        }
        public void UpdatePlayer(Player player,string id)
        {
            if (playerpainter == null)
            {
                player.Id = id;
                playerpainter = new PlayerPainter(this, player);
                painters.Add("playerpainter",playerpainter);
            }
            else
            {
                playerpainter.Update(player);
            }
            mappainter.Center = playerpainter.Position;
        }

        //private void UpdatePlayer(object sender, object e)
        //{

        //    if (playerpainter == null)
        //    {
        //        playerpainter = new PlayerPainter(this, e.ID, (int)e.X, (int)e.Y, (double)e.Dir, e.Actions);
        //        painters.Add(playerpainter);
        //    }
        //    else
        //    {
        //        playerpainter.Update(e.X, e.Y, e.Dir, e.Actions);
        //    }
        //    mappainter.Center = playerpainter.Position;
        //}


        private void Tick(object sender, TickEventArgs e)
        {
            var pts = painters.Values.ToArray();
            foreach (var painter in pts)
            {
                painter.Draw();
            }
            screen.Blit(buffer);
            screen.Update();
        }

        void Close(object sender, QuitEventArgs e)
        {
            Events.QuitApplication();
        }
    }
}
