﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace RunningGame
{
    public class Game
    {
        GameForm gameForm;
        Player player1;
        Player player2;
        Stopwatch watch;
        int finishLine;
        int countdownTime;
        bool finished;
        bool started;
        
        // Properties for multithreading and cancellation of tasks
        CancellationTokenSource drawLookTs;
        CancellationTokenSource logicLookTs;
        Task drawLoop;
        Task logicLoop;

        // Constants
        const int LOGIC_LOOP_INTERVAL = 50;

        #region Getters and Setters

        public bool Started
        {
            get { return started; }
            set { started = value; }
        }

        public int CountdownTime
        {
            get { return countdownTime; }
            set { countdownTime = value; }
        }

        #endregion

        public Game(GameForm gameForm, Player player1, Player player2, int distance)
        {
            this.gameForm = gameForm;
            this.player1 = player1;
            this.player2 = player2;
            this.player1.TimeTillImgChange = LOGIC_LOOP_INTERVAL * 5;
            this.player2.TimeTillImgChange = LOGIC_LOOP_INTERVAL * 5;
            finishLine = distance;

            finished = false;
            started = false;
            countdownTime = 3000;

            StartGame();
        }

        public void StartGame()
        {
            // Start logic loop as a new task
            logicLookTs = new CancellationTokenSource();
            logicLoop = new Task(() =>
            {
                // Reduce the countdownTime before the race starts
                while (!started)
                {
                    countdownTime = countdownTime - LOGIC_LOOP_INTERVAL;
                    if (countdownTime <= 0) started = true;
                    Thread.Sleep(LOGIC_LOOP_INTERVAL);
                }

                // Adapt player speeds and positions after the race has started
                while (!finished && started)
                {
                    AdaptSpeed(player1);
                    CheckBlockProtection(player1);
                    AdaptSpeed(player2);
                    CheckBlockProtection(player2);
                    Thread.Sleep(LOGIC_LOOP_INTERVAL); // Update speeds every 50 ms
                }

            }, logicLookTs.Token);
            logicLoop.Start();

            // Start draw loop as a new task
            drawLookTs = new CancellationTokenSource();
            drawLoop = new Task(() => { while (!finished) { gameForm.UpdateGameForm(); } }, drawLookTs.Token);
            drawLoop.Start();

            // Start stopwatch
            watch = new Stopwatch();
            watch.Start();
        }

        /// <summary>
        /// Process every key input and change a player's speed and state accordingly
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void ProcessKeyInput(object sender, KeyEventArgs e)
        {
            // Player 1 makes first step
            if ((e.KeyCode == Keys.A || e.KeyCode == Keys.S) && player1.State == PlayerState.Standing)
            {
                player1.Speed += 3;
                player1.State = (e.KeyCode == Keys.A) ? PlayerState.StepLeft : PlayerState.StepRight;
            }

            // Player 2 makes first step
            if ((e.KeyCode == Keys.K || e.KeyCode == Keys.L) && player2.State == PlayerState.Standing)
            {
                player2.Speed += 3;
                player2.State = (e.KeyCode == Keys.K) ? PlayerState.StepLeft : PlayerState.StepRight;
            }

            // Player 1 keeps running and makes step left
            if (e.KeyCode == Keys.A && player1.State == PlayerState.StepRight)
            {
                player1.Speed += 3;
                player1.State = PlayerState.StepLeft;
            }

            // Player 1 keeps running and makes step right
            if (e.KeyCode == Keys.S && player1.State == PlayerState.StepLeft)
            {
                player1.Speed += 3;
                player1.State = PlayerState.StepRight;
            }

            // Player 2 keeps running and makes step left
            if (e.KeyCode == Keys.K && player2.State == PlayerState.StepRight)
            {
                player2.Speed += 3;
                player2.State = PlayerState.StepLeft;
            }

            // Player 2 keeps running and makes step right
            if (e.KeyCode == Keys.L && player2.State == PlayerState.StepLeft)
            {
                player2.Speed += 3;
                player2.State = PlayerState.StepRight;
            }

            // Player 1 executes a block on Player 2
            if (e.KeyCode == Keys.Q)
                ExecuteBlockOn(player1, player2);

            // Player 2 executes a block on Player 1
            if (e.KeyCode == Keys.I)
                ExecuteBlockOn(player2, player1);

            // Player 1 activates block protection
            if (e.KeyCode == Keys.W)
                ActivateBlockProtection(player1);

            // Player 2 activates block protection
            if (e.KeyCode == Keys.O)
                ActivateBlockProtection(player2);
        }

        /// <summary>
        /// Reset the opponents speed to zero
        /// </summary>
        /// <param name="blocking"></param>
        /// <param name="blocked"></param>
        private void ExecuteBlockOn(Player blocking, Player blocked)
        {
            if (blocking.BlockEnabled && !blocked.BlockProtectionActivated)
            {
                blocked.Speed = 0;
                blocking.BlockEnabled = false;
            }
        }

        /// <summary>
        /// Activate a block protection one's own player for two seconds (2000 ms)
        /// </summary>
        /// <param name="player"></param>
        private void ActivateBlockProtection(Player player)
        {
            if (player.BlockProtectionEnabled)
            {
                player.BlockProtectionActivated = true;
                player.BlockProtectionEnabled = false;
            }
        }

        /// <summary>
        /// Check the current state of block protection and end protection if blockProtectionTime has been exceeded
        /// </summary>
        /// <param name="player"></param>
        private void CheckBlockProtection(Player player)
        {
            if (player.BlockProtectionActivated && player.BlockProtectionTime > 0)
                player.BlockProtectionTime -= LOGIC_LOOP_INTERVAL;
            else
            {
                player.BlockProtectionTime = 2000;
                player.BlockProtectionActivated = false;
            }
        }

        /// <summary>
        /// Reduce speed with every logic loop or set speed to zero as a minimum
        /// </summary>
        private void AdaptSpeed(Player player)
        {
            if (player.Speed > 45)
                player.Speed -= 2;
            else if (player.Speed > 0)
                player.Speed--;
            else player.Speed = 0;

            AdaptX(player);
            AdaptImage(player);

            if (HasFinished(player))
                EndGame(player);
        }

        /// <summary>
        /// Modify the runner's x-Coordinate according to its speed
        /// </summary>
        /// <param name="player"></param>
        private void AdaptX(Player player)
        {
            int speed = player.Speed;

            if (speed == 0)
            {
                player.ImgChangeInterval = 0;
            }
            else if (speed < 3)
            {
                player.XCoordinate += 2;
                player.ImgChangeInterval = 1;
            }
            else if (speed < 9)
            {
                player.XCoordinate += 4;
                player.ImgChangeInterval = 2;
            }
            else if (speed < 15)
            {
                player.XCoordinate += 6;
                player.ImgChangeInterval = 3;
            }
            else if (speed < 18)
            {
                player.XCoordinate += 8;
                player.ImgChangeInterval = 4;
            }
            else
            {
                player.XCoordinate += 10;
                player.ImgChangeInterval = 5;
            }
        }

        /// <summary>
        /// Set the the image of the running character (referenced by an index)
        /// </summary>
        /// <param name="player"></param>
        private void AdaptImage(Player player)
        {
            int changeInterval = player.ImgChangeInterval;
            int timeTillChange = player.TimeTillImgChange;

            // A highter changeInterval means a higher speed and a faster image change
            switch (changeInterval)
            {
                case 5:
                    timeTillChange -= LOGIC_LOOP_INTERVAL * 5;
                    break;
                case 4:
                    timeTillChange -= LOGIC_LOOP_INTERVAL * 4;
                    break;
                case 3:
                    timeTillChange -= LOGIC_LOOP_INTERVAL * 3;
                    break;
                case 2:
                    timeTillChange -= LOGIC_LOOP_INTERVAL * 2;
                    break;
                case 1:
                    timeTillChange -= LOGIC_LOOP_INTERVAL * 1;
                    break;
                default:
                    timeTillChange -= LOGIC_LOOP_INTERVAL * 0;
                    break;
            }

            if (timeTillChange <= 0)
            {
                if (player.PlayerImageIndex == player.Character.Sprites - 1)
                {
                    // Start picture of the image row
                    player.PlayerImageIndex = 0;
                }
                else
                {
                    // Iterating through the image row (->motion)
                    player.PlayerImageIndex++;
                }
                timeTillChange = LOGIC_LOOP_INTERVAL * 5;
            }
            player.TimeTillImgChange = timeTillChange;
        }

        /// <summary>
        /// Check if one of the player's has already reached the finish line
        /// </summary>
        /// <param name="player"></param>
        /// <returns></returns>
        bool HasFinished(Player player)
        {
            return player.XCoordinate >= finishLine - player.Character.SpriteHeight / 3;
        }

        /// <summary>
        /// Stop the stopwatch, cancel all runnings tasks (draw and logic loop) and show the result as a message box
        /// </summary>
        /// <param name="player"></param>
        void EndGame(Player player)
        {
            finished = true;

            drawLookTs.Cancel();
            logicLookTs.Cancel();

            watch.Stop();

            string message = player.Name + " has won the race in " + (watch.ElapsedMilliseconds * 1.0) / 1000 + " seconds.\nDo you want to restart?";
            string caption = "Game over";
           
            if (MessageBox.Show(message, caption, MessageBoxButtons.YesNo) == DialogResult.Yes)
                gameForm.Invoke(new Action(() => gameForm.Close()), null);
            else
                Application.Exit();
        }
    }
}