﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace RunningGame
{
    public partial class GameForm : Form
    {
        // Properties for game logic
        Game game;
        Player player1;
        Player player2;
        int distance;

        // Properties for design
        SettingsForm settingsForm;
        Bitmap bmp;
        Bitmap startlightsBmp;
        Rectangle leftLight;
        Rectangle middleLight;
        Rectangle rightLight;

        // Properties for synchronization
        LabelHandler handler;
        SynchronizationContext ctx;

        // Properties for handling different Screens/DPIs
        SizeF currentScaleFactor = new SizeF(1f, 1f);

        # region Getters and setters

        public Game Game
        {
            get { return game; }
            set { game = value; }
        }

        public Label SpeedPlayer1Label
        {
            get { return speedPlayer1Label; }
            set { speedPlayer1Label = value; }
        }

        public Label SpeedPlayer2Label
        {
            get { return speedPlayer2Label; }
            set { speedPlayer2Label = value; }
        }

        public Label PositionPlayer1Label
        {
            get { return positionPlayer1Label; }
            set { positionPlayer1Label = value; }
        }

        public Label PositionPlayer2Label
        {
            get { return positionPlayer2Label; }
            set { positionPlayer2Label = value; }
        }

        # endregion

        public GameForm(SettingsForm settingsForm, Player player1, Player player2)
        {
            InitializeComponent();

            // Get synchronization context for the current UI thread
            ctx = SynchronizationContext.Current;

            // Set delegate to invoke control components asynchronously
            handler = new LabelHandler(SetLabelText);

            // Set players and race distance
            this.player1 = player1;
            this.player2 = player2;
            distance = 2000;

            // Show player names with labels
            namePlayer1Label.Text = player1.Name;
            namePlayer2Label.Text = player2.Name;

            // Enable double buffering
            SetStyle(ControlStyles.OptimizedDoubleBuffer | ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint, true);

            // Enable access to the settings form; needed for restarting the game
            this.settingsForm = settingsForm;

            // Draw initial game form
            DrawTrack(pictureBoxPlayer1, player1, player1.Character.SpriteSheet);
            DrawTrack(pictureBoxPlayer2, player2, player2.Character.SpriteSheet);

            // Set "Action" icons to correct resolution
            AdjustActionIconsResolution();      

            // Initialize a new game instance
            game = new Game(this, player1, player2, distance);
        }

        /// <summary>
        /// Delegate key inputs (after the game has started)
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Form1_KeyUp(object sender, KeyEventArgs e)
        {
            if (game.Started)
                game.ProcessKeyInput(sender, e);
        }

        /// <summary>
        /// Invokes drawing methods for the traffic lights and/or game screens
        /// </summary>
        public void UpdateGameForm()
        {
            if (!game.Started)
                DrawStartLights();
            else
            {
                DrawStartLights();
                DrawTrack(pictureBoxPlayer1, player1, player1.Character.SpriteSheet);
                DrawTrack(pictureBoxPlayer2, player2, player2.Character.SpriteSheet);
            }
        }

        /// <summary>
        /// Invoke draw methods for single lights and set its colors depending on the countdown timer
        /// </summary>
        private void DrawStartLights()
        {
            float width = this.currentScaleFactor.Width;
            float height = this.currentScaleFactor.Height;

            int startlightsBmpWidth = (int)(156 * width);
            int startlightsBmpHeight = (int)(52 * height);

            startlightsBmp = new Bitmap(startlightsBmpWidth, startlightsBmpHeight);

            int lightWidth = (int)(46 * width);
            int lightHeight = (int)(46 * height);

            int xValueSecondLight = (int)(55 * width);
            int xValueThirdLight = (int)(107 * width);

            leftLight = new Rectangle(3, 3, lightWidth, lightHeight);
            middleLight = new Rectangle(xValueSecondLight, 3, lightWidth, lightHeight);
            rightLight = new Rectangle(xValueThirdLight, 3, lightWidth, lightHeight);

            using (Graphics g = Graphics.FromImage(startlightsBmp))
            {
                // After the game has started, start lights stay green
                if (game.Started)
                    DrawLights(g, Brushes.Green, Brushes.Green, Brushes.Green);

                // Fill background
                g.Clear(Color.Black);

                // Draw traffic lights
                if (game.CountdownTime >= 2000)
                    DrawLights(g, Brushes.Red, Brushes.Gray, Brushes.Gray);
                else if (game.CountdownTime >= 1000 && game.CountdownTime < 2000)
                    DrawLights(g, Brushes.Red, Brushes.Red, Brushes.Gray);
                else if (game.CountdownTime > 0 && game.CountdownTime < 1000)
                    DrawLights(g, Brushes.Red, Brushes.Red, Brushes.Red);
                else
                    DrawLights(g, Brushes.Green, Brushes.Green, Brushes.Green);

                // Update the traffic lights
                pbStartLights.Invoke(new Action(() => pbStartLights.Image = startlightsBmp), null);
            }
        }

        /// <summary>
        /// Draw the traffic lights
        /// </summary>
        /// <param name="g"></param>
        /// <param name="colorLeft"></param>
        /// <param name="colorMiddle"></param>
        /// <param name="colorRight"></param>
        private void DrawLights(Graphics g, Brush colorLeft, Brush colorMiddle, Brush colorRight)
        {
            g.DrawEllipse(new Pen(colorLeft), leftLight.X, leftLight.Y,
               leftLight.Width, leftLight.Height);
            g.FillEllipse(colorLeft, leftLight.X, leftLight.Y,
                leftLight.Width, leftLight.Height);
            g.DrawEllipse(new Pen(colorMiddle), middleLight.X, middleLight.Y,
                middleLight.Width, middleLight.Height);
            g.FillEllipse(colorMiddle, middleLight.X, middleLight.Y,
                middleLight.Width, middleLight.Height);
            g.DrawEllipse(new Pen(colorRight), rightLight.X, rightLight.Y,
                rightLight.Width, rightLight.Height);
            g.FillEllipse(colorRight, rightLight.X, rightLight.Y,
                rightLight.Width, rightLight.Height);
        }

        private void DrawTrack(PictureBox pb, Player player, Bitmap src)
        {
            float width = this.currentScaleFactor.Width;
            float height = this.currentScaleFactor.Height;

            int trackBmpWidth = (int)(884 * width);
            int trackBmpHeight = (int)(200 * height);

            bmp = new Bitmap(trackBmpWidth, trackBmpHeight);
       
            Rectangle targetRect = new Rectangle(getDistanceFromTarget(player) + (pb.Width / 3), 130, 5, 40);

            using (Graphics graphics = Graphics.FromImage(bmp))
            {
                // Draw moving background
                if ((player.XCoordinate > pb.Width / 3) && (player.XCoordinate < (pb.Width + (pb.Width / 3))))
                {
                    graphics.DrawImage(
                        Resources.background_sky, (player.XCoordinate - pb.Width / 3) * (-1) , 0,
                        Resources.background_sky.Width * this.currentScaleFactor.Width,
                        Resources.background_sky.Height * this.currentScaleFactor.Height
                        );
                    graphics.DrawImage(
                        Resources.background_sky, pb.Width - (player.XCoordinate - pb.Width / 3), 0,
                        Resources.background_sky.Width * this.currentScaleFactor.Width,
                        Resources.background_sky.Height * this.currentScaleFactor.Height
                        );
                    graphics.DrawRectangle(new Pen(Brushes.Red), targetRect);
                    graphics.FillRectangle(Brushes.Red, targetRect);
                }
                else if (player.XCoordinate > (pb.Width + (pb.Width / 3)))
                {
                    graphics.DrawImage(
                        Resources.background_sky, (player.XCoordinate - (pb.Width + (pb.Width / 3))) * (-1), 0,
                        Resources.background_sky.Width * this.currentScaleFactor.Width,
                        Resources.background_sky.Height * this.currentScaleFactor.Height
                        );
                    graphics.DrawImage(
                        Resources.background_sky, pb.Width - (player.XCoordinate - (pb.Width + (pb.Width / 3))), 0,
                        Resources.background_sky.Width * this.currentScaleFactor.Width,
                        Resources.background_sky.Height * this.currentScaleFactor.Height
                        );
                    graphics.DrawRectangle(new Pen(Brushes.Red), targetRect);
                    graphics.FillRectangle(Brushes.Red, targetRect);
                }
                else
                {
                    graphics.DrawImage(
                        Resources.background_sky, 0, 0,
                        Resources.background_sky.Width * this.currentScaleFactor.Width,
                        Resources.background_sky.Height * this.currentScaleFactor.Height
                        );
                    graphics.DrawRectangle(new Pen(Brushes.Red), targetRect);
                    graphics.FillRectangle(Brushes.Red, targetRect);
                }

                // Draw sprite section
                Bitmap sprite = new Bitmap(player.Character.SpriteWidth, player.Character.SpriteHeight);
                Graphics g = Graphics.FromImage(sprite);
                Rectangle spriteSection = GetSpriteSection(player);
                g.DrawImage(src, new Rectangle(0, 0, player.Character.SpriteWidth, player.Character.SpriteHeight), spriteSection, GraphicsUnit.Pixel);

                if (player.XCoordinate > pb.Width / 3)
                {
                    Rectangle rect = new Rectangle(pb.Width / 3, trackBmpHeight / 2, (int) (player.Character.SpriteWidth * width), (int) (player.Character.SpriteHeight * height));
                    graphics.DrawImage(sprite, rect);
                }
                else
                {
                    Rectangle rect = new Rectangle(player.XCoordinate, trackBmpHeight / 2, (int)(player.Character.SpriteWidth * width), (int)(player.Character.SpriteHeight * height));
                    graphics.DrawImage(sprite, rect);
                }

                pb.Image = bmp;
            }

            // Update labels
            UpdateLabelText(SpeedPlayer1Label, player1.Speed.ToString());
            UpdateLabelText(SpeedPlayer2Label, player2.Speed.ToString());

            if (player1.XCoordinate > player2.XCoordinate)
            {
                UpdateLabelText(PositionPlayer1Label, "1/2");
                UpdateLabelText(PositionPlayer2Label, "2/2");
            }
            else
            {
                UpdateLabelText(PositionPlayer1Label, "2/2");
                UpdateLabelText(PositionPlayer2Label, "1/2");
            }

            ctx.Post(o => blockProtectionEnabledPlayer1PictureBox.Visible = player1.BlockProtectionEnabled, null);
            ctx.Post(o => blockProtectionEnabledPlayer2PictureBox.Visible = player2.BlockProtectionEnabled, null);
            ctx.Post(o => blockProtectionActivatedPlayer1PictureBox.Visible = player1.BlockProtectionActivated, null);
            ctx.Post(o => blockProtectionActivatedPlayer2PictureBox.Visible = player2.BlockProtectionActivated, null);
            ctx.Post(o => blockEnabledPlayer1PictureBox.Visible = player1.BlockEnabled, null);
            ctx.Post(o => blockEnabledPlayer2PictureBox.Visible = player2.BlockEnabled, null);
        }

        private int getDistanceFromTarget(Player player)
        {
            return distance - player.XCoordinate;
        }

        /// <summary>
        /// Get a single sprite section from original picture
        /// </summary>
        /// <param name="player"></param>
        /// <returns></returns>
        private Rectangle GetSpriteSection(Player player)
        {
            int index = player.PlayerImageIndex;
            int width = player.Character.SpriteWidth;
            int height = player.Character.SpriteHeight;

            Rectangle rect;
            switch (index)
            {
                case 0:
                    rect = new Rectangle(0, 0, width, height);
                    break;
                case 1:
                    rect = new Rectangle(width, 0, width, height);
                    break;
                case 2:
                    rect = new Rectangle(width * 2, 0, width, height);
                    break;
                case 3:
                    rect = new Rectangle(width * 3, 0, width, height);
                    break;
                case 4:
                    rect = new Rectangle(width * 4, 0, width, height);
                    break;
                case 5:
                    rect = new Rectangle(width * 5, 0, width, height);
                    break;
                case 6:
                    rect = new Rectangle(width * 6, 0, width, height);
                    break;
                default:
                    rect = new Rectangle(width, 0, width, height);
                    break;
            }
            return rect;
        }

        #region One delegate and two methods for asynchronously updating the labels in the game form

        private delegate void LabelHandler(Label label, String text);

        private void UpdateLabelText(Label label, string text)
        {
            if (label.InvokeRequired)
                label.Invoke(handler, new object[] { label, text });
            else
                SetLabelText(label, text);
        }

        private void SetLabelText(Label label, string text)
        {
            label.Text = text;
        }

        #endregion

        /// <summary>
        /// Show settings form once the game form is closed and show options to restart/quit the game
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void GameForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            settingsForm.Invoke(new Action(() => settingsForm.Show()), null);
        }

        /// <summary>
        /// Overriding scale control method to have the same scaling on monitors with different DPIs
        /// </summary>
        /// <param name="factor"></param>
        /// <param name="specified"></param>
        protected override void ScaleControl(SizeF factor, BoundsSpecified specified)
        {
            base.ScaleControl(factor, specified);

            // record the running scale factor used
            this.currentScaleFactor = new SizeF(
                this.currentScaleFactor.Width * factor.Width,
                this.currentScaleFactor.Height * factor.Height);
        }

        private void AdjustActionIconsResolution()
        {
            float width = this.currentScaleFactor.Width;
            float height = this.currentScaleFactor.Height;

            blockEnabledPlayer1PictureBox.Image = new Bitmap(Resources.symbol_block, new Size(
                (int)Math.Round(Resources.symbol_block.Width * width),
                (int)Math.Round(Resources.symbol_block.Width * height)));
            blockEnabledPlayer2PictureBox.Image = new Bitmap(Resources.symbol_block, new Size(
                (int)Math.Round(Resources.symbol_block.Width * width),
                (int)Math.Round(Resources.symbol_block.Height * height)));
            blockProtectionActivatedPlayer1PictureBox.Image = new Bitmap(Resources.symbol_protection, new Size(
                (int)Math.Round(Resources.symbol_protection.Width * width),
                (int)Math.Round(Resources.symbol_protection.Height * height)));
            blockProtectionActivatedPlayer2PictureBox.Image = new Bitmap(Resources.symbol_protection, new Size(
                (int)Math.Round(Resources.symbol_protection.Width * width),
                (int)Math.Round(Resources.symbol_protection.Height * height)));
            blockProtectionEnabledPlayer1PictureBox.Image = new Bitmap(Resources.symbol_shield, new Size(
                (int)Math.Round(Resources.symbol_shield.Width * width),
                (int)Math.Round(Resources.symbol_shield.Height * height)));
            blockProtectionEnabledPlayer2PictureBox.Image = new Bitmap(Resources.symbol_shield, new Size(
                (int)Math.Round(Resources.symbol_shield.Width * width),
                (int)Math.Round(Resources.symbol_shield.Height * height)));
        }
    }
}
