﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.Windows.Forms;
using System.Xml.Serialization;
using System.IO;
using System.Diagnostics;


namespace Sudoku
{
    public partial class MainForm : Form
    {
        byte[,] array = new byte[Globals.SIZE, Globals.SIZE];
        SudokuSolver sudoku;
        bool changed = false;                       // Gibt an, ob die Lösung neu berechnet werden muss
        int baseSeverity = (int)(Globals.TOTAL / 3.5);

        List<Point> labels = new List<Point>();     // Listet die Stellen im Feld, an denen Labels statt Textboxen stehen
        List<Point> hints = new List<Point>();      // Listet die Stellen im Feld, an denen ein Tipp gegeben wurde
        int hintsCount { get { return hints.ToArray().Length; } }


        // Konstruktor Vorgabe selbst eintragen

        internal MainForm()
        {
            InitializeComponent();
            initializeWithSIZE();

            helpTextChangedEvent();

            this.Text = "Vorgabe festlegen";
            ok.Visible = true;
            hint.Visible = false;
            solve.Visible = false;
            testSolve.Visible = false;
            save.Visible = false;
        }

        // Konstruktor Vorgabe generieren lassen

        internal MainForm(SudokuSolver generated, Severity level)
        {
            InitializeComponent();
            initializeWithSIZE();

            helpTextChangedEvent();

            sudoku = generated;
            Random rand = new Random();
            
            for (int i = 0; i < (((int)level) * Globals.BOXSIZE + baseSeverity); i++)
            {
                Control c = getRandEmptyC(rand);
                c.Text = "" + sudoku.Field[field.GetRow(c), field.GetColumn(c)];
            }
            inputToLabel();
        }

        // Konstruktor laden

        internal MainForm(byte[,] array, List<Point> labels, List<Point> hints)
        {
            InitializeComponent();
            initializeWithSIZE();

            helpTextChangedEvent();

            foreach (Point posOfLabel in labels)
                field.GetControlFromPosition(posOfLabel.Y, posOfLabel.X).Text = "" + array[posOfLabel.X, posOfLabel.Y];

            inputToLabel();

            foreach (Point posOfLabel in hints)
            {
                field.GetControlFromPosition(posOfLabel.Y, posOfLabel.X).BackColor = Color.Yellow;
                this.hints.Add(posOfLabel);
            }

            for (int i = 0; i < Globals.SIZE; i++)
                for (int j = 0; j < Globals.SIZE; j++)
                    if (field.GetControlFromPosition(j, i).Text == "" && array[i, j] != 0)
                    {
                        field.GetControlFromPosition(j, i).Text = "" + array[i, j];
                    }
            changed = true;
        }


        // Event für beliebige Eingaben erzeugen

        private void helpTextChangedEvent()
        {
            for (int i = 0; i < Globals.SIZE; i++)
                for (int j = 0; j < Globals.SIZE; j++)
                {
                    var textbox = field.GetControlFromPosition(i, j) as TextBox;

                    if (textbox != null)
                    {
                        textbox.TextChanged += textbox_TextChanged;
                    }
                }
        }

        async private void textbox_TextChanged(object sender, EventArgs e)
        {
            buttonEnabler(true);
            message.Visible = false;
            error.Visible = false;
            ok.Enabled = true;
            changed = true;
            try 
            {
                read();
                if (textFull())
                {
                    await sudokuAsyncSolver();
                    progressBar.Visible = false;
                    buttonEnabler(true);           
                    MessageBox.Show("Herzlichen Glückwunsch!\nSie haben das Sudoku richtig gelöst.\nBenötigte Tipps: " + hintsCount+".");
                }
            }
            catch (InvalidValueExeption ex)
            {
                error.Visible = true;
                error.Text = ex.message;

                buttonEnabler(false);
                ok.Enabled = false;
            }
            catch (NotSolveableExeption ex)
            {
                error.Visible = true;
                error.Text = ex.message3;

                buttonEnabler(false);
                ok.Enabled = false;
                progressBar.Visible = false;
            }
        }


        // Events der Buttons (Ok-Button nur in Form zum Vorgabe eingeben sichtbar)

        async private void ok_Click(object sender, EventArgs e)
        {
            ok.Enabled = false;
            try
            {
                await sudokuAsyncSolver();
                progressBar.Visible = false;
                inputToLabel();
                this.Text = "Sudoku - Viel Spaß!";
                ok.Visible = false;
                hint.Visible = true;
                solve.Visible = true;
                testSolve.Visible = true;
                buttonEnabler(true);
                save.Visible = true;
            }
            catch (NotSolveableExeption ex)
            {
                error.Visible = true;
                error.Text = ex.message1;

                progressBar.Visible = false;
                ok.Enabled = false;
            }
        }

        async private void hint_Click(object sender, EventArgs e)
        {
            try
            {
                if (changed)
                {
                    await sudokuAsyncSolver();
                    progressBar.Visible = false;
                    buttonEnabler(true);
                    changed = false;
                }

                Random rand = new Random();
                Control c = getRandEmptyC(rand);
                if ( c!= null)
                {
                    byte h = sudoku.Field[field.GetRow(c), field.GetColumn(c)];
                    c.Text = "" + h;
                    hints.Add(new Point(field.GetRow(c), field.GetColumn(c)));
                    c.BackColor = Color.Yellow;
                }
                changed = false;
            }
            catch (NotSolveableExeption ex)
            {
                error.Visible = true;
                error.Text = ex.message2;

                buttonEnabler(false);
                progressBar.Visible = false;
            }
        }

        async private void testSolve_Click(object sender, EventArgs e)
        {
            try
            {
                if (changed)
                {
                    await sudokuAsyncSolver();
                    progressBar.Visible = false;
                    buttonEnabler(true);
                    changed = false;
                }
                message.Visible = true;
            }
            catch (NotSolveableExeption ex)
            {
                error.Visible = true;
                error.Text = ex.message2;

                buttonEnabler(false); ;
                progressBar.Visible = false;
            }
        }

        async private void solve_Click(object sender, EventArgs e)
        {
            try
            {
                if (changed)
                {
                    await sudokuAsyncSolver();
                    progressBar.Visible = false;
                    buttonEnabler(true);
                    changed = false;
                }
                SolutionForm window = new SolutionForm(field, sudoku.Field);
                window.Show();
            }
            catch (NotSolveableExeption ex)
            {
                error.Visible = true;
                error.Text = ex.message2;

                buttonEnabler(false);
                progressBar.Visible = false;
            }
            catch (ObjectDisposedException)
            {
            }
        }

        public void save_Click(object sender, EventArgs e)
        {
            try
            {
                read();
            }
            catch (InvalidValueExeption)
            {
            }
            SudokuInfos info = new SudokuInfos(array, labels, hints);
            info.Save();
        }

        private void quit_Click(object sender, EventArgs e)
        {
            sudoku = null;
            this.Close();
        }


        // Hilfsfunktionen

        private void read()
        {
            byte value;
            byte i, j;
            Control c;

            for (i = 0; i < Globals.SIZE; i++)
                for (j = 0; j < Globals.SIZE; j++)
                {
                    c = field.GetControlFromPosition(j, i);
                    if ((byte.TryParse(c.Text, out value) || c.Text == "") && value <= Globals.SIZE)
                        array.SetValue(value, new int[] {field.GetRow(c), field.GetColumn(c)});
                    else
                        throw new InvalidValueExeption();
               }
        }

        private void inputToLabel()
        {
            Control c;

            for (int i = 0; i < Globals.SIZE; i++)
                for (int j = 0; j < Globals.SIZE; j++)
                {
                    c = field.GetControlFromPosition(j, i);
                    if (c != null)
                    {
                        string text;
                        string name;
                        int row, column;

                        if (c.Text != "")
                        {
                            name = c.Name;
                            text = c.Text;
                            row = field.GetRow(c);
                            column = field.GetColumn(c);
                            field.Controls.Remove(c);
                            c.Dispose();

                            Label label = new Label();
                            label.Text = text;
                            label.Size = new System.Drawing.Size(19, 19);
                            label.BackColor = Color.Transparent;
                            field.Controls.Add(label, column, row);
                            label.Dock = DockStyle.Bottom;
                            label.Name = name;
                            labels.Add(new Point(row, column));
                        }
                    }
                }
            changed = false;
        }

        private Task sudokuAsyncSolver()
        {
            SynchronizationContext sync = SynchronizationContext.Current;
            int progress = 0;

            progressBar.Visible = true;
            buttonEnabler(false);
            return Task.Run(() =>
            {
                sudoku = new SudokuSolver(array, sync, () =>
                {
                    progress++;
                    progressBar.Value = (int)(Math.Tanh((Math.Sqrt(progress) / 10)) * 100);
                });
            });
        }

        private bool textFull()
        {
            Control c;

            for (int i = 0; i < Globals.SIZE; i++)
                for (int j = 0; j < Globals.SIZE; j++)
                {
                   c = field.GetControlFromPosition(j, i);
                   if (c.Text == "")
                       return false;
                }
            return true;
        }

        private Control getRandEmptyC(Random rand)
        {
            List<Control> emptyC = new List<Control>();
            Control c = null;

            for (int i = 0; i < Globals.SIZE; i++)
                for (int j = 0; j < Globals.SIZE; j++)
                {
                    c = field.GetControlFromPosition(j, i);
                    if (c.Text == "")
                        emptyC.Add(c);
                }
            if (emptyC.ToArray().Length != 0)
            {
                c = emptyC[rand.Next(emptyC.ToArray().Length)];
            }
            return c;
        }

        private void buttonEnabler(bool enable)
        {
            hint.Enabled = enable;
            solve.Enabled = enable;
            testSolve.Enabled = enable;
        }

        private void field_Paint(object sender, PaintEventArgs e)
        {
            Graphics g = e.Graphics;
            Pen pen = new Pen(Color.Gray, 2f);

            for (int i = 0; i < Globals.BOXSIZE; i++)
                for (int j = 0; j < Globals.BOXSIZE; j++)
                {
                    g.DrawRectangle(pen, new Rectangle(i * Globals.BOXSIZE * 25, j * Globals.BOXSIZE * 25, Globals.BOXSIZE * 25, Globals.BOXSIZE * 25));
                }
            g.DrawRectangle(pen, new Rectangle(1, 1, Globals.SIZE * 25 - 2, Globals.SIZE * 25 - 2));
        }

        private void initializeWithSIZE()
        {
            field.ColumnCount = Globals.SIZE;
            field.RowCount = Globals.SIZE;
            this.field.Size = new System.Drawing.Size(Globals.SIZE * 25, Globals.SIZE * 25);

            for (int i = 0; i < Globals.SIZE; i++)
                for (int j = 0; j < Globals.SIZE; j++)
                {
                    System.Windows.Forms.TextBox textbox = new System.Windows.Forms.TextBox();
                    textbox.Name = "textBox" + i + j;
                    this.field.Controls.Add(textbox, i, j);
                    textbox.Size = new System.Drawing.Size(19, 20);
                    textbox.TabIndex = (i + 1) + Globals.SIZE * j;
                }

            int h = Globals.SIZE * 25 + 81 - 50 - 23 - 29;          // Höhe des OK- / Save-Buttons
            if (h < 122 + 29) h = 122 + 29;

            solve.Location = new System.Drawing.Point(Globals.SIZE * 25 + 50, 93);
            hint.Location = new System.Drawing.Point(Globals.SIZE * 25 + 50, 35);
            testSolve.Location = new System.Drawing.Point(Globals.SIZE * 25 + 50, 64);
            save.Location = new System.Drawing.Point(Globals.SIZE * 25 + 50, h);
            quit.Location = new System.Drawing.Point(Globals.SIZE * 25 + 50, h + 29);
            ok.Location = new System.Drawing.Point(Globals.SIZE * 25 + 50, h);
            progressBar.Location = new System.Drawing.Point(Globals.SIZE * 25 + 50, 122);

            this.Size = new System.Drawing.Size(Globals.SIZE * 25 + 104 + 100, Globals.SIZE * 25 + 81);

            this.solve.TabIndex = Globals.TOTAL + 3;
            this.hint.TabIndex = Globals.TOTAL + 1;
            this.testSolve.TabIndex = Globals.TOTAL + 2;
            this.save.TabIndex = Globals.TOTAL + 4;
            this.quit.TabIndex = Globals.TOTAL + 5;
            this.ok.TabIndex = Globals.TOTAL + 4;
        }
    }
}