var GameObject = function(x, y) {
	this.setPosition(x, y);
	this.setVelocity();
	this.setDimension();
};

GameObject.prototype.setPosition = function(x, y) {
	this.x = x || 0;
	this.y = y || 0;
};

GameObject.prototype.setVelocity = function(vx, vy) {
	this.vx = vx || 0;
	this.vy = vy || 0;

};

GameObject.prototype.setDimension = function(width, height) {
	this.width = width || 0;
	this.height = height || 0;
};

GameObject.prototype.collidesWith = function(that) {

	if (this.top() < that.bottom() && this.bottom() > that.top() &&
		this.right() > that.left()  && this.left() < that.right() ) {
		return true;
	}
};

GameObject.prototype.left = function() {
	return this.x;
};

GameObject.prototype.top = function() {
	return this.y;
};

GameObject.prototype.right = function() {
	return this.x + this.width;
};

GameObject.prototype.bottom = function() {
	return this.y + this.height;
};

GameObject.prototype.step = function(game) {
	this.x += this.vx;
	this.y += this.vy;
};



var Game = function(width, height) {
	this.setPosition(0, 0);
	this.setDimension(width, height);
	this.objects = [];
	this.players = [];
	this.pipes = [];
	this.time = 0;
	this.stepTime = 15;
	this.loop = undefined;
	this.pipeCountDown = 150;
	this.killed = false;
		
	this.objects.push(new Grass(this.width*100000, height / 4, 0, 400));
	this.objects.push(new Grass(this.width*100000, 1, 0, -20));
}

Game.prototype = new GameObject();

Game.prototype.addPlayer = function(bird) {
	this.players.push(bird);
	this.objects.push(bird);
	bird.setControls();
};

Game.prototype.addPipe = function() {
	
	for (var i = this.objects.length - 1; i >= 0; i--) {
		if (this.objects[i] instanceof Pipe) {
			minX = this.objects[i].right();
			break;
		}
	};

	var x = this.x + this.width;
	var ho = Math.floor(Math.random() * 200) + 20; //Pipeoben-länge zw. 20 und 200
	var l = Math.floor(Math.random() * 60) + 40; //Lückenbreite random zw. 40 und 60
	var hu = this.height - ho - l - 80; //Pipeunten-länge
	var pipeUp = new Pipe(x, 0, 30, ho);
	var pipeDown = new Pipe(x, ho + l, 30, hu);

	this.objects.push(pipeUp, pipeDown);
	this.pipes.push(pipeUp, pipeDown);
};

Game.prototype.step = function() {
	this.time++;
	this.x++;
	this.pipeCountDown--;

	if (this.pipeCountDown === 0) {
		this.pipeCountDown = 150;
		this.addPipe();
	}

	for (var i = this.objects.length - 1; i >= 0; i--)
		this.objects[i].step(this);
};

Game.prototype.run = function(ctx) {
	var me = this;
	me.loop = setInterval(function() {
		me.step();
	}, me.stepTime);

	var drawing = function() {
		me.draw(ctx);
		window.requestAnimationFrame(drawing);
	};

	drawing();
};

Game.prototype.pause = function() {
	clearInterval(this.loop);
	this.loop = undefined;
	
};

Game.prototype.killPlayer = function(bird) {
	this.pause();
	this.killed = true;	
	
};

var birdend = 0;


Game.prototype.draw = function(ctx) {
	var width = this.width;
	var height = this.height;
	ctx.clearRect(0, 0, width, height);

	ctx.save();
	ctx.translate(-this.x, 0);

	for (var i = this.objects.length - 1; i >= 0; i--)
		this.objects[i].draw(ctx);
		ctx.restore();


	if (this.killed === false) {
		//Bird stirbt nicht >> Punkte zählen
		ctx.font ="20pt Arial bold";
		ctx.fillStyle = 'red';
		text = this.players[0].getScore();
		ctx.fillText(text, 10, 30 );
		localStorage.setItem("now", text);
	

	}else{ //Game over und Punkte mit Highscore zeigen
		ctx.fillStyle = 'green';
		ctx.fillRect(100,70,180,40);
		ctx.font ="20pt Arial bold";
		ctx.fillStyle = 'white';
		ctx.fillText('GAME OVER', 110, 100 );

		ctx.font ="15pt Arial";
		ctx.fillStyle = 'white';
	

		var text1 = 'Better your best!';
		var text2 = 'Score: ' + this.players[0].getScore();
		var text3= 'Best:'+ best;

		ctx.fillText(text3, 220, 430 );
		ctx.fillText(text2, 120, 430 );
		ctx.fillText(text1, 130, 460 );

		hscore = Math.max(last, text);
		localStorage.setItem("last", hscore);

	};
	
		last = localStorage.getItem("last") || 0;
		now = localStorage.getItem("now") || 0;
		best = Math.max(now, last) || 0;
		
};



var Bird = function(x, y, resources) { 
	this.setPosition(x, y);
	this.setDimension(20, 20);
	this.birdImage = resources.bird;
	this.spriteIndex = 0;
	this.passedPipes = [];
	this.falling = true;
	this.controls = {};
};


Bird.prototype = new GameObject();

Bird.prototype.setControls = function() {
	var bird = this;

	document.body.addEventListener('touchstart', function(e){
			bird.vy = -1.5;
			bird.vx = 1;
			bird.falling = false;

	});

	document.body.addEventListener('touchend', function(e){
			bird.vy = 1;
			bird.vx = 1;
			bird.falling = true;

	});

	this.controls.keydown = function(e) {
		switch (e.keyCode) {
			case 38:
				bird.vy = -1.5;
				break;
		}

		bird.falling = false;
	};

	this.controls.keyup = function(e) {
		bird.vx = 1;
		bird.vy = 1;
		bird.falling = true;
	};

	this.controls.mousedown = function(e) {
		bird.vy = -1.5;
		bird.vx = 1;
		bird.falling = false;
	};

	this.controls.mouseup = function(e) {
		bird.vy = 1;
		bird.vx = 1;
		bird.falling = true;
	};
};

Bird.prototype.getScore = function() {
	score = this.passedPipes.length / 2;
	return score;
};

Bird.prototype.step = function(game) {
	if (this.falling) {
		this.vy = this.vy + .05;
	}

	GameObject.prototype.step.apply(this, [game]);

	if (game.time % 10 === 0)
		this.spriteIndex = (this.spriteIndex + 1) % 3;

	for (var i = game.pipes.length; i--; ) {
		var pipe = game.pipes[i];

		if (pipe.right() < this.left() && this.passedPipes.indexOf(pipe) === -1) {
			this.passedPipes.push(pipe);
		}

	};

};


Bird.prototype.draw = function(ctx) {

	ctx.drawImage(this.birdImage, 0, 22 * this.spriteIndex, 25, 20, this.x, this.y, this.width, this.height);
};

var Pipe = function(x, y, width, height) {
	this.setPosition(x, y);
	this.setDimension(width + 15 || 0, height || 0);
};

Pipe.prototype = new GameObject();

Pipe.prototype.draw = function(ctx) {

	ctx.fillStyle = "blue"; 
	ctx.fillRect(this.x, this.y, this.width + 5, this.height);
};

Pipe.prototype.step = function(game) {
	for (var j = game.players.length - 1; j >= 0; j--) {
		var bird = game.players[j];

		if (bird.collidesWith(this))
			game.killPlayer(bird);
	}
};

var Grass = function(width, height, xPosition, yPosition) {
	this.setPosition(xPosition, yPosition);
	this.setDimension(width, height);
};

Grass.prototype = new GameObject();

Grass.prototype.step = function(game) {
	for (var j = game.players.length - 1; j >= 0; j--) {
		var bird = game.players[j];

		if (bird.collidesWith(this)) {
			game.killPlayer(bird);
		}
	}
};

Grass.prototype.draw = function(ctx) {
	ctx.fillStyle = "green";
	ctx.fillRect(this.x, this.y, this.width, this.height);
	
};

//-----------------------------------------------------------

function loadResource(resources, resourceLoaded, resource) {
	var element = document.createElement('img');
	element.src = resource.source;
	element.addEventListener('load', function() {
		resources[resource.name] = element;
		resourceLoaded();
	});
}

function loadResources(resourcesLoaded) {
	var resources = {};
	var images = [{ 
		name: 'bird', 
		source: 'sprite.png'
	}];
	var total = images.length;

	for (var i = images.length; i--; ) {
		loadResource(resources, function() {
			if (--total === 0)
				resourcesLoaded();
		}, images[i]);
	}
	
	return resources;

}

function setControls(game) {
	for (var i = game.players.length; i--; ) {
		var player = game.players[i];
		var controlKeys = Object.keys(player.controls);

		for (var j = controlKeys.length; j--; ) {
			var key = controlKeys[j];
			window.addEventListener(key, player.controls[key]);
		}
	}
}

function unsetControls(game) {
	for (var i = game.players.length; i--; ) {
		var player = game.players[i];
		var controlKeys = Object.keys(player.controls);

		for (var j = controlKeys.length; j--; ) {
			var key = controlKeys[j];
			window.removeEventListener(key, player.controls[key]);
		}
	}
}

var game = undefined;
var resources = undefined;

function startGame() {
	if (game !== undefined)
		return;

	var width = 400;
	var height = 480;
	var canvas = document.querySelector("canvas");
	var ctx = canvas.getContext("2d");
	canvas.width = width;
	canvas.height = height;	
	ctx.clearRect(0, 0, canvas.width, canvas.height);
	game = new Game(width, height);
	var bird = new Bird(150, 100, resources);
	game.addPlayer(bird);
	setControls(game);
	game.addPipe();	
	game.run(ctx);
	
}

function endGame() {
	if (game === undefined)
		return;

	game.pause();
	unsetControls(game);
	game = undefined;
	
};

function main() {
	resources = loadResources(function() {
		startGame();

	});		

document.querySelector('#playgame').addEventListener('click', function(ev) {

		endGame();
		startGame();
	});
};








