﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

namespace Projekt
{
    public abstract class Car : Vehicle
    {
        protected IContext context;

        #region Wheels
        protected ISteeringWheel frontLeft;
        protected ISteeringWheel frontRight;
        protected IDrivingWheel rearLeft;
        protected IDrivingWheel rearRight;

        public ISteeringWheel FrontLeft
        {
            get { return frontLeft; }
        }

        public ISteeringWheel FrontRight
        {
            get { return frontRight; }
        }

        public IDrivingWheel RearLeft
        {
            get { return rearLeft; }
        }

        public IDrivingWheel RearRight
        {
            get { return rearRight; }
        }
        #endregion

        #region Methods

        /// <summary>
        /// Calculates and updates the current placement, velocity and acceleration of the car
        /// </summary>
        public override void Update()
        {
            var HalfIntSqu = context.Interval_s * context.Interval_s / 2.0;

            var normalForceRear = context.GravitationAcceleration * Mass * ((Length - CenterOfMassHorizontal)) / (2.0 * Length); 
            var normalForceFront = context.GravitationAcceleration * Mass * CenterOfMassHorizontal / (2.0 * Length);

            var forceFrontLeft = FrontLeft.GetForce(normalForceFront, Velocity);
            var forceFrontRight = FrontRight.GetForce(normalForceFront, Velocity);
            var forceRearLeft = RearLeft.GetForce(normalForceRear, Velocity);
            var forceRearRight = RearRight.GetForce(normalForceRear, Velocity);

            var steeringAngle = FrontLeft.CurrentAngle;
            var w2 = Width / 2.0;
            var angleChange = 0.0;
            var torque = (forceRearRight - forceRearLeft) * w2;

            if ((Velocity * CurrentPower.Watt).SignificantlySmallerZero())
            {
                torque += (forceFrontRight - forceFrontLeft) * w2;
            }
            else if (steeringAngle.Rad.SignificantlyNotZero())
            {
                Double sina = steeringAngle.sin();
                Double w2cosa = steeringAngle.cos(w2);

                if (steeringAngle.Rad < Math.PI)
                {
                    torque += sina * (Length - CenterOfMassHorizontal) * (forceFrontRight + forceFrontLeft) + w2cosa * (forceFrontRight - forceFrontLeft);
                }
                else
                {
                    torque += sina * (Length - CenterOfMassHorizontal) * (forceFrontRight + forceFrontLeft) - w2cosa * (forceFrontRight - forceFrontLeft);
                }
                angleChange += Velocity * context.Interval_s * Math.Sin(steeringAngle.Rad) / Length;
            }

            angleChange += HalfIntSqu * torque * 12.0 / (Mass * (Length * Length + Width * Width));
            Angle newAngle = (Angle)(Position.Alignment + angleChange);

            var totalForceForward = forceRearLeft + forceRearRight;
            totalForceForward += (forceFrontLeft * Velocity).SignificantlyGreaterZero() ? FrontLeft.CurrentAngle.cos(forceFrontLeft) : forceFrontLeft;
            totalForceForward += (forceFrontRight * Velocity).SignificantlyGreaterZero() ? FrontRight.CurrentAngle.cos(forceFrontRight) : forceFrontRight;
            totalForceForward -= DragCoefficient * context.AirDensity * Velocity * Math.Abs(Velocity);
            Acceleration = totalForceForward / Mass;
            var changeForward = HalfIntSqu * Acceleration + context.Interval_s * Velocity;
            
            if (changeForward.NotZero())
            {
                var changeX = Position.Alignment.cos(changeForward);
                var changeY = Position.Alignment.sin(changeForward);

                Velocity = Math.Sqrt(changeX * changeX + changeY * changeY) / context.Interval_s;
                Velocity *= changeForward > 0.0 ? 1.0 : -1.0;
                Position = new Placement(Position.Coordinates.X + changeX, Position.Coordinates.Y - changeY, newAngle);
            }
            else
            {
                Velocity = 0.0;
                Position = new Placement(Position.Coordinates.X, Position.Coordinates.Y, newAngle);
            }
            SetWheels();
        }

        /// <summary>
        /// Sets torque and angles of the wheels according to the control
        /// </summary>
        /// <param name="control">control of the vehicle</param>
        public override void Control(IVehicleControl control)
        {
            if (control.IsAccelerating && !control.IsDecelerating)
            {
                if (Velocity.SmallerZero())
                {
                    CurrentPower = MaxPower;
                }
                else
                {
                    CurrentPower = Power.Min(CurrentPower + MaxPower * 0.01, MaxPower);
                }
            }
            else if (control.IsDecelerating && !control.IsAccelerating)
            {
                if (Velocity.GreaterZero())
                {
                    CurrentPower = -MaxPower;
                }
                else
                {
                    CurrentPower = Power.Max(CurrentPower - MaxPower * 0.01, -MaxPower);
                }
            }
            else
            {
                CurrentPower = new Power(0.0);
            }

            ApplyTorque();

            var angle = frontLeft.CurrentAngle;
            var max = frontLeft.MaxAngle;

            if (control.IsTurningLeft && !control.IsTurningRight)
            {
                ApplyAngle(Angle.Min((angle.Rad > Math.PI ? (Angle)0.0 : angle) + max / (10.0 * (1.0 + Math.Abs(Velocity))), max));
            }
            else if (control.IsTurningRight && !control.IsTurningLeft)
            {
                ApplyAngle(Angle.Max((angle.Rad > Math.PI ? angle : (Angle)0.0) - max / (10.0 * (1.0 + Math.Abs(Velocity))), -max));
            }
            else
            {
                if (Math.Abs(angle.Degpm) < 3)
                {
                    ApplyAngle((Angle)0.0);
                }
                else
                {
                    ApplyAngle(Angle.FromDeg(Math.Sign(angle.Radpm) * (Math.Abs(angle.Degpm) - 1.0)));
                }
            }
        }

        /// <summary>
        /// Calculates and assigns the position of/to each wheel, depending on the placement of the car
        /// </summary>
        protected override void SetWheels()
        {
            var centerWheelDist = Math.Sqrt(Length * Length + Width * Width) / 2.0;
            var carX = Position.Coordinates.X;
            var carY = Position.Coordinates.Y;

            var fixedAngle = (Angle)(Math.Asin(Width / (2.0 * centerWheelDist)));

            var wheelAngle = (Angle)(fixedAngle + Position.Alignment);
            var sin = wheelAngle.sin(centerWheelDist);
            var cos = wheelAngle.cos(centerWheelDist);

            FrontLeft.Position = new Coord(carX + cos, carY - sin);
            RearRight.Position = new Coord(carX - cos, carY + sin);

            wheelAngle = (Angle)(Position.Alignment - fixedAngle);
            sin = wheelAngle.sin(centerWheelDist);
            cos = wheelAngle.cos(centerWheelDist);

            FrontRight.Position = new Coord(carX + cos, carY - sin);
            RearLeft.Position = new Coord(carX - cos, carY + sin);
        }

        /// <summary>
        /// The important information about the current status of the car
        /// </summary>
        /// <returns>important information of the car</returns>
        public override String ToString()
        {
            return String.Format("x: {0}\ny: {1}\nangle: {2:000.}°\ntorque (Nm):\n   rl: {3:0000.}\n   rr: {4:0000.}\n   fl: {5:0000.}\n   fr: {6:0000.}\nvelocity: {7:0.00} m/s\nacceleration: {8:0.0000} m/s^2\npower: {9} HP\nsteering angle: {10}°", (Int32)Position.X, (Int32)Position.Y, (Int32)Position.Alignment.Deg, (Int32)rearLeft.Torque, (Int32)rearRight.Torque, (Int32)frontLeft.Torque, (Int32)frontRight.Torque, Velocity, Acceleration, (Int32)CurrentPower.HP, (Int32)frontLeft.CurrentAngle.Degpm);
        }
        
        #endregion
    }
}
