﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Gapi.Search;
using System.Threading;
using System.Windows.Forms;
using System.Diagnostics;

namespace QuizSome
{
    public class Quiz
    {
        public GameState currentState = GameState.login;
        public event EventHandler<myGameStateEvent> stateChanged;
        bool loggedIn = false;
        List<User> players;
        List<List<Questions>> gameQuestions;
        public int CurrentPlayerNumber { get; private set; }
        public List<int> Dices { get; private set; }
        public List<int> Points { get; set; }
        public List<long> Times { get; set; }
        public int CurrentRound { get; private set; }
        Stopwatch sw;
        public List<List<bool>> Joker { get; set; }
        public int MaximumRounds { get; set; }

        /// <summary>
        /// Constructor which initalizes all variables
        /// </summary>
        public Quiz()
        {
            var r = new Random();
            gameQuestions = new List<List<Questions>>()
            {
                XmlParserQuestion.loadQuestions().OrderBy(m => r.Next()).Take(11).ToList(),
                XmlParserQuestion.loadQuestions().OrderBy(m => r.Next()).Skip(11).Take(11).ToList()
            };
            Dices = new List<int>();
            Points = new List<int>() { 0, 0 };
            Times = new List<long>() { 0, 0 };
            CurrentRound = 0;
            sw = new Stopwatch();
            Joker = new List<List<bool>>()
            {
                new List<bool>() {true, true, true},
                new List<bool>() {true, true, true}
            };
            MaximumRounds = 7;
        }

        /// <summary>
        /// Get the score of a random user.
        /// </summary>
        /// <param name="i">Player one or two (so 0 or 1)</param>
        /// <returns>Score of a user.</returns>
        public int GetScore(int i)
        {
            return players[i].Score;
        }

        /// <summary>
        /// Get the won games of a random user.
        /// </summary>
        /// <param name="i">Player one or two (so 0 or 1)</param>
        /// <returns>Wins of a user.</returns>
        public int GetWins(int i)
        {
            return players[i].Won;
        }

        /// <summary>
        /// Get the lost games of a random user.
        /// </summary>
        /// <param name="i">Player one or two (so 0 or 1)</param>
        /// <returns>Losts of a user.</returns>
        public int GetLosts(int i)
        {
            return players[i].Lost;
        }

        /// <summary>
        /// Get a random user name.
        /// </summary>
        /// <param name="i">Player one or two (so 0 or 1)</param>
        /// <returns>Name of a user.</returns>
        public string GetUserName(int i)
        {
            return players[i].Nickname;
        }

        /// <summary>
        /// Updates the player stats.
        /// </summary>
        public void UpdatePlayer()
        {
            players[0].Score += Points[0];
            players[1].Score += Points[1];
            if (Points[0] > Points[1])
            {
                players[0].Won++;
                players[1].Lost++;
            }
            else if (Points[0] < Points[1])
            {
                players[1].Won++;
                players[0].Lost++;
            }
            else
            {
                if (Times[0] > Times[1])
                {
                    players[0].Won++;
                    players[1].Lost++;
                }
                else
                {
                    players[1].Won++;
                    players[0].Lost++;
                }
            }
        }

        /// <summary>
        /// Get the current user name.
        /// </summary>
        /// <returns>Returns the current user whose turn it is.</returns>
        public string GetUserName()
        {
            return players[CurrentPlayerNumber].Nickname;
        }

        /// <summary>
        /// Signs the users in, throws the dices and shows the dicing window.
        /// </summary>
        /// <param name="nick1">nickname player one</param>
        /// <param name="pwd1">password player one</param>
        /// <param name="nick2">nickname player two</param>
        /// <param name="pwd2">password player two</param>
        public void SignIn(string nick1, string pwd1, string nick2, string pwd2)
        {
            players = new List<User>() 
            {
                XmlParserUser.Login(nick1, pwd1),
                XmlParserUser.Login(nick2, pwd2)
            };
            loggedIn = (players[0] != null && players[1] != null);
            if (loggedIn)
            {
                ThrowDice();
                currentState = GameState.beforeDicing;
                if (stateChanged != null)
                    stateChanged(this, new myGameStateEvent(currentState));
            }
            else
            {
                currentState = GameState.wrongLogin;
                if (stateChanged != null)
                    stateChanged(this, new myGameStateEvent(currentState));
            }
        }

        /// <summary>
        /// This methods changes questions according to the shift joker.
        /// </summary>
        public void ChangeQuestions()
        {
            if(CurrentRound < MaximumRounds-1)
            {
                if (CurrentPlayerNumber == 0)
                {
                    gameQuestions[1][CurrentRound] = gameQuestions[0][CurrentRound];
                    gameQuestions[0][CurrentRound] = gameQuestions[0][MaximumRounds];
                    Joker[0][2] = false;
                }
                if (CurrentPlayerNumber == 1)
                {
                    gameQuestions[0][CurrentRound + 1] = gameQuestions[1][CurrentRound];
                    gameQuestions[1][CurrentRound] = gameQuestions[1][MaximumRounds];
                    Joker[1][2] = false;
                }
            }
        }

        /// <summary>
        /// Gets the current question for the current round and current player.
        /// </summary>
        /// <returns></returns>
        public Questions GetCurrentQuestion()
        {
            return gameQuestions[CurrentPlayerNumber][CurrentRound];
        }

        /// <summary>
        /// Changes the game state.
        /// </summary>
        /// <param name="state">Takes a current game state.</param>
        public void ChangeState(GameState state)
        {
            currentState = state;
            if (stateChanged != null)
                stateChanged(this, new myGameStateEvent(currentState));
        }

        /// <summary>
        /// By restarting the game, the dice has to thrown again. This happens with this method.
        /// </summary>
        public void RestartDicing()
        {
            var r = new Random();
            int p1 = 0;
            int p2 = 0;
            while (p1 == p2)
            {
                p1 = r.Next(1, 6);
                p2 = r.Next(1, 6);
            }
            Dices[0] = p1;
            Dices[1] = p2;
            if (p1 > p2)
                CurrentPlayerNumber = 0;
            else
                CurrentPlayerNumber = 1;
        }

        /// <summary>
        /// Creates to random variables and saves them internaly. They will be printed out after the dice is
        /// thrown in the game.
        /// </summary>
        void ThrowDice()
        {
            var r = new Random();
            int p1 = 0;
            int p2 = 0;
            while (p1 == p2)
            {
                p1 = r.Next(1, 6);
                p2 = r.Next(1, 6);
            }
            if (p1 > p2)
                CurrentPlayerNumber = 0;
            else
                CurrentPlayerNumber = 1;
            Dices.Add(p1);
            Dices.Add(p2);
            if (stateChanged != null)
                stateChanged(this, new myGameStateEvent(currentState));
        }

        /// <summary>
        /// Starts the game.
        /// </summary>
        public void Start()
        {
            currentState = GameState.gaming;
            if (stateChanged != null)
                stateChanged(this, new myGameStateEvent(currentState));
        }

        /// <summary>
        /// Proofs if the user is still logged in. If so the user will be logged out.
        /// </summary>
        public void UserLogout()
        {
            if (loggedIn)
            {
                XmlParserUser.logout(players[0]);
                XmlParserUser.logout(players[1]);
                players[0] = null;
                players[1] = null;
                currentState = GameState.loggedOut;
                if (stateChanged != null)
                    stateChanged(this, new myGameStateEvent(currentState));
                loggedIn = false;
            }
        }

        /// <summary>
        /// Quits the game. Saves the players stats and changes the game state.
        /// </summary>
        public void Quit()
        {
            UserLogout();
            currentState = GameState.quit;
            if (stateChanged != null)
                stateChanged(this, new myGameStateEvent(currentState));
        }

        int InternalRoundCounter = 0;
        /// <summary>
        /// Checks the given answer and continues the game.
        /// </summary>
        /// <param name="p">An given answer (string).</param>
        internal void checkIfAnswerTrue(string p)
        {
            sw.Stop();
            var question = GetCurrentQuestion();
            var answers = question.Answers;
            var n = p;
            var singleSolution = answers.Where(m => { return m.Question == n; }).FirstOrDefault();

            if (singleSolution.IsTrue)
            {
                Points[CurrentPlayerNumber]++;
            }

            InternalRoundCounter++;
            if (InternalRoundCounter % 2 == 0)
                CurrentRound++;

            if (CurrentRound == MaximumRounds)
            {
                currentState = GameState.finished;
                if (stateChanged != null)
                    stateChanged(this, new myGameStateEvent(currentState));
            }

            Times[CurrentPlayerNumber] += sw.Elapsed.Milliseconds;

            CurrentPlayerNumber = (CurrentPlayerNumber+1)%2;
            sw.Restart();
        }

        /// <summary>
        /// Starts the stopwatch.
        /// </summary>
        internal void StartStopWatch()
        {
            sw.Start();
        }

        /// <summary>
        /// Resets the whole game for a restart.
        /// </summary>
        internal void ResetAll()
        {
            sw.Stop();
            sw.Reset();
            Points[0] = 0;
            Points[1] = 0;
            Times[0] = 0;
            Times[1] = 0;
            CurrentRound = 0;
        }

        /// <summary>
        /// Mixes the questions so the players can not have the same question(s) in one game.
        /// </summary>
        internal void MixQuestions()
        {
            var r = new Random();
            gameQuestions = new List<List<Questions>>()
            {
                XmlParserQuestion.loadQuestions().OrderBy(m => r.Next()).Take(MaximumRounds+1).ToList(),
                XmlParserQuestion.loadQuestions().OrderBy(m => r.Next()).Take(MaximumRounds+1).ToList()
            };
        }

        /// <summary>
        /// Updates a current questions. Is used for the fifty fifty joker.
        /// </summary>
        internal void UpdateQuestions()
        {
            var question = GetCurrentQuestion();
            var answers = question.Answers;
            var singleSolution = answers.Where(m => m.IsTrue == false).First();
            singleSolution.Question = "";
            var singleSolution2 = answers.Where(m => m.IsTrue == false && m.Question != "").First();
            singleSolution2.Question = "";
            Joker[CurrentPlayerNumber][0] = false;
        }
    }
}
