﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Gapi.Search;

namespace QuizSome
{
    public partial class QuizScreenControl : UserControl
    {
        public Quiz CurrentGame { get; set; }
        List<Button> AnswerButtons;
        List<Button> JokerButtons;

        /// <summary>
        /// Initalizes the quiz screen and shows it up.
        /// </summary>
        public QuizScreenControl()
        {
            InitializeComponent();
            AnswerButtons = new List<Button>() { answerA, answerB, answerC, answerD };
            JokerButtons = new List<Button>() { fifty, google, shift };
        }

        /// <summary>
        /// Sets everything in the game you need to play: points, score, names, etc. It also adds a check function for the answer
        /// buttons.
        /// </summary>
        /// <param name="game">variable of the class game</param>
        internal void recieveGame(Quiz game)
        {
            CurrentGame = game;
            SetPointsAndTime();
            SetNameAndScore();
            SetQuestion();
            for (int i = 0; i < 4; i++)
                AnswerButtons[i].Click += CheckIfClickedButtonWasTheRightAnswer;

            fifty.Visible = true;
            google.Visible = true;
            shift.Visible = true;
        }

        /// <summary>
        /// This deletes questions on the screen. (50:50 joker)
        /// </summary>
        public void DeleteQuestions()
        {
            for (int i = 0; i < 4; i++)
                AnswerButtons[i].Click -= CheckIfClickedButtonWasTheRightAnswer;
            EmptyQuestion();
        }

        /// <summary>
        /// Sets the current question to the games screen.
        /// </summary>
        void SetQuestion()
        {
            var q = CurrentGame.GetCurrentQuestion();
            question.Text = q.Ask;
            for (int i = 0; i < 4; i++)
            {
                AnswerButtons[i].Text = q.Answers[i].Question;
            }
        }

        /// <summary>
        /// Deletes the question and it answers on the screen.
        /// </summary>
        public void EmptyQuestion()
        {
            question.Text = "";
            for (int i = 0; i < 4; i++)
            {
                AnswerButtons[i].Text = "";
            }
        }

        void CheckIfClickedButtonWasTheRightAnswer(object o, EventArgs e)
        {
            CurrentGame.checkIfAnswerTrue(((Button)o).Text);
            if(CurrentGame.currentState == GameState.gaming)
                SetQuestion();
            SetPointsAndTime();
        }

        /// <summary>
        /// Sets joker, player names, player points and players elapsed time. Also prints who's turn it is.
        /// </summary>
        void SetPointsAndTime()
        {
            p1Points.Text = CurrentGame.Points[0].ToString() + " points";
            p2Points.Text = CurrentGame.Points[1].ToString() + " points";
            p1Time.Text = CurrentGame.Times[0].ToString()+" ms";
            p2Time.Text = CurrentGame.Times[1].ToString() + " ms";
            whichPlayer.Text = CurrentGame.GetUserName().ToString();
            google.Visible = CurrentGame.Joker[CurrentGame.CurrentPlayerNumber][1];
            shift.Visible = CurrentGame.Joker[CurrentGame.CurrentPlayerNumber][2];
            fifty.Visible = CurrentGame.Joker[CurrentGame.CurrentPlayerNumber][0];
        }


        /// <summary>
        /// Prints the stats every player has.
        /// </summary>
        public void SetNameAndScore()
        {
            setPlayerName1.Text = CurrentGame.GetUserName(0) + " (Score: " + CurrentGame.GetScore(0) + ", Wins: " + CurrentGame.GetWins(0) + ", Lost: " + CurrentGame.GetLosts(0) + ")";
            setPlayerName2.Text = CurrentGame.GetUserName(1) + " (Score: " + CurrentGame.GetScore(1) + ", Wins: " + CurrentGame.GetWins(1) + ", Lost: " + CurrentGame.GetLosts(1) + ")";
        }


        /// <summary>
        /// A Google Solver which solves a current questions depending on a google query. If no connection can be established this joker returns 25% for every answer.
        /// </summary>
        void GoogleSolver()
        {
            List<long> results = new List<long>(4);
            for (int i = 0; i < 4; i++)
            {
                try
                {
                    var query = CurrentGame.GetCurrentQuestion();
                    var queryString = query.Ask + " " + query.Answers[i].Question;
                    var x = Searcher.Search(SearchType.Web, queryString);
                    results.Add(x.EstimatedResultCount);
                }
                catch
                {
                    results.Add(25);
                }
            }
            var sum = results.Sum();
            var solution = new List<int>();
            for (int i = 0; i < results.Count; i++)
            {
                solution.Add((int)((((float)results[i]) / sum) * 100));
            }
            CurrentGame.Joker[CurrentGame.CurrentPlayerNumber][1] = false;
            SetPointsAndTime();
            labelA.Width = solution[0] * 2;
            labelB.Width = solution[1] * 2;
            labelC.Width = solution[2] * 2;
            labelD.Width = solution[3] * 2;
            labelA.Text = solution[0] + "%";
            labelB.Text = solution[1] + "%";
            labelC.Text = solution[2] + "%";
            labelD.Text = solution[3] + "%";
        }

        /// <summary>
        /// The google joker.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void google_Click(object sender, EventArgs e)
        {
            GoogleSolver();
        }

        /// <summary>
        /// The fifty fifty joker.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void fifty_Click(object sender, EventArgs e)
        {
            CurrentGame.UpdateQuestions();
            fifty.Visible = CurrentGame.Joker[CurrentGame.CurrentPlayerNumber][0];
            SetQuestion();
        }

        /// <summary>
        /// The shift joker.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void shift_Click(object sender, EventArgs e)
        {
            CurrentGame.ChangeQuestions();
            shift.Visible = CurrentGame.Joker[CurrentGame.CurrentPlayerNumber][2];
            SetQuestion();
        }
    }
}
